use core:io;
use sql;

DATABASE ProgvisDB {
	// Users in the system.
	TABLE users(
		// User id. Used in other tables.
		id INTEGER PRIMARY KEY,
		// User name.
		name TEXT ALLOW NULL,
		// User display name.
		displayName TEXT,
		// User is administrator?
		admin INTEGER DEFAULT 0
	);

	// Known clients in the system, and what users they map to.
	TABLE clients(
		// Client ID (a long string).
		id TEXT PRIMARY KEY UNIQUE,
		// User ID.
		user INTEGER
	);

	// Problems in the system. Each problem corresponds to an initial problem submitted by some
	// user. This initial submission contains an implementation of some data structure, a test for
	// the data structure, and a reference implementation. Users will then improve the
	// implementation and the test, so they are expected to be "poor" initially.
	TABLE problems(
		// ID of this problem.
		id INTEGER PRIMARY KEY,
		// Problem author (foreign key to users)
		author INTEGER,
		// Title of the problem.
		title TEXT,
		// Implementation (foreign key to code)
		impl INTEGER,
		// Test (foreign key to code)
		test INTEGER,
		// Reference implementation (foreign key to code)
		refimpl INTEGER,
		// Created (time string in UTC)
		created TEXT,
		// Checks to enable.
		checks TEXT DEFAULT "",
		// Acceptable errors.
		acceptableErrors TEXT DEFAULT ""
	);

	// Improvements to implementations.
	TABLE implementations(
		// ID of this implementation.
		id INTEGER PRIMARY KEY,
		// Problem ID (foreign to problems).
		problem INTEGER,
		// Version of the implementation (increasing for each user+problem), ID:s are not guaranteed to be increasing on SQLite.
		version INTEGER DEFAULT 1,
		// Author.
		author INTEGER,
		// Code (foreign to code).
		code INTEGER,
		// Which test was used to highlight an error here? (foreign to tests, -1 if original test)
		basedOn INTEGER,
		// Created.
		created TEXT
	);
	INDEX ON implementations(author);

	// Improvements to tests.
	TABLE tests(
		// ID of this test.
		id INTEGER PRIMARY KEY,
		// Problem ID (foreign to problems).
		problem INTEGER,
		// Version of the test (increasing for each user+problem), ID:s are not guaranteed to be increasing on SQLite.
		version INTEGER DEFAULT 1,
		// Author.
		author INTEGER,
		// Code (foreign to code).
		code INTEGER,
		// Which implementation was used to find errors in? (foreign to implementations, -1 if original impl.)
		basedOn INTEGER,
		// Created.
		created TEXT
	);
	INDEX ON tests(author);

	// Known combinations of success/failure between tests.
	// We don't aim to test all combinations, but only the latest version of each users' submission.
	TABLE autoresults(
		problem INTEGER,
		// Implementation tested (implementations table). -1 means the original implementation.
		impl INTEGER,
		// Test used (tests table). -1 means the original test.
		test INTEGER,
		// Any errors found? String is type of error found. NULL means no error was found.
		error TEXT ALLOW NULL,
		// PK declaration.
		PRIMARY KEY(problem, impl, test)
	);

	// Code stored in the system.
	TABLE code(
		// ID of this code.
		id INTEGER PRIMARY KEY,
		// Program source code.
		src TEXT,
		// Language (= file extension).
		language TEXT
	);

	// Log attempts at finding bugs in the code.
	TABLE attempts(
		// ID of the sequence.
		id INTEGER PRIMARY KEY,
		// Author.
		author INTEGER,
		// ID of the implementation used (refers to code).
		impl INTEGER,
		// ID of the test used (refers to code).
		test INTEGER,
		// Sequence of advanced threads.
		sequence TEXT,
		// Error found at the end, if any.
		error TEXT ALLOW NULL,
		// Timestamp.
		created TEXT
	);

}

class Database {
	init() {
		Url baseDir = cwdUrl;
		Url subDir = baseDir / "db";
		if (subDir.exists)
			baseDir = subDir;

		SQLite db(baseDir / "progvis.db");

		init() {
			db(db);
		}
	}

	private ProgvisDB db;

	// Find a user's identity from its client key.
	UserInfo? findUser(Str clientId) {
		if (x = WITH db: SELECT ONE users.id, users.displayName, users.admin FROM clients JOIN users ON clients.user == users.id WHERE clients.id == clientId) {
			return UserInfo(x.users.id, x.users.displayName, x.users.admin > 0);
		}
		null;
	}

	// Find user info from its ID.
	UserInfo? findUser(Int userId) {
		if (x = WITH db: SELECT ONE users.id, users.displayName, users.admin FROM users WHERE users.id == userId) {
			return UserInfo(x.users.id, x.users.displayName, x.users.admin > 0);
		}
		null;
	}

	// Find a user's name from its ID.
	Str? findUserName(Int userId) {
		if (x = WITH db: SELECT ONE displayName FROM users WHERE id == userId) {
			return x.displayName;
		} else {
			return null;
		}
	}

	// Log out a client.
	void logout(Str clientId) {
		WITH db: DELETE FROM clients WHERE id == clientId;
	}

	// Change username.
	void changeName(Int userId, Str newName) {
		WITH db: UPDATE users SET displayName = newName WHERE id == userId;
	}

	// Create a new problem.
	Int createProblem(Int userId, Str title, Code impl, Code test, Code refImpl,
					ProblemChecks checks, AcceptableErrors acceptable, Error error) {
		Int implId = createCode(impl);
		Int testId = createCode(test);
		Int refId = createCode(refImpl);
		Int problemId = WITH db: INSERT INTO
			problems(author, title, impl, test, refimpl, created, checks, acceptableErrors)
			VALUES (userId, title, implId, testId, refId, CURRENT DATETIME, ${checks.toDB()}, ${acceptable.toDB()});

		addError(problemId, -1, -1, error);

		return problemId;
	}

	// Store a sequence of an attempted "solution"
	Int postAttempt(Int userId, Int implId, Int testId, Str sequence, Str? error) {
		checkImplId(implId);
		checkTestId(testId);

		WITH db: INSERT INTO attempts(author, impl, test, sequence, error, created)
			VALUES (userId, implId, testId, sequence, error, CURRENT DATETIME);
	}

	// Compute the points of all users in the database. Returns map of userId -> score.
	Int->Int allScores() {
		// Scores:
		// Points given to a user who...
		// ...solves an initial problem. Only awarded if enough issues are solved.
		Int initialSubmitFull = 50;
		Int initialSubmitError = 40;
		// ...submits an improvement for a debug problem (i.e. a new implementation).
		Int submittedImpl = 10;
		// ...submits an improvement to a debug problem that solves a previous error.
		Int solvedImpl = 50;
		// ...submits an improved test (these need to solve an error somewhere).
		Int submittedTest = 100;

		// Divider if interacting with one's own problems.
		Int selfPenalty = 3;

		Int->Int result;
		WITH db {
			// Check implementations:
			for (row in SELECT i.id AS iId, i.author AS iAuthor, i.problem AS problem, t.id AS tId, t.author AS tAuthor
					 FROM implementations i
					 JOIN tests t ON t.id == i.basedOn) {
				Int points = submittedImpl;

				// Note: We enforce that the previous setup must have contained an error, so we don't need to check that.
				Error e = findError(row.problem, row.iId, row.tId);
				if (e.success())
					points += solvedImpl;

				if (row.iAuthor == row.tAuthor)
					points /= selfPenalty;

				result[row.iAuthor] += points;
			}

			// Check initial implementations:
			for (row in SELECT id, author, problem FROM implementations WHERE basedOn == -1) {
				// Note: We can do this with a nice join.
				// Note: Some of these may contain errors, if they have not solved the problem "well enough".
				if (findError(row.problem, row.id, -1).success())
					result[row.author] += initialSubmitFull;
				else
					result[row.author] += initialSubmitError;
			}

			// Check tests:
			for (row in SELECT t.id AS newId, t.author AS tAuthor, t.problem AS problem, i.id AS iId, i.author AS iAuthor
					 FROM tests t
					 JOIN implementations i ON i.id == t.basedOn) {
				Int points = submittedTest;

				if (row.iAuthor == row.tAuthor)
					points /= selfPenalty;

				result[row.tAuthor] += points;
			}

			// Check initial tests:
			for (row in SELECT author FROM tests WHERE basedOn == -1) {
				result[row.author] += submittedTest;
			}
		}
		return result;
	}

	// Get a list of all problems in the system.
	ProblemInfo[] problemList(Int userId) {
		ProblemInfo[] result;
		WITH db {
			for (row in SELECT p.id AS id, u.displayName AS author, p.title AS title
					 FROM problems p
					 JOIN users u ON p.author == u.id) {
				Bool attempted =
					(SELECT ONE id FROM implementations WHERE problem == ${row.id} AND author == userId).any |
					(SELECT ONE id FROM tests WHERE problem == ${row.id} AND author == userId).any;
				Error error = findError(row.id, -1, -1);
				result << ProblemInfo(row.id, row.title, row.author, error.any(), attempted);
			}
		}
		result;
	}

	// Get a list of all problems in the system. Don't populate 'attempted' as we have no user.
	ProblemInfo[] problemList() {
		ProblemInfo[] result;
		WITH db {
			for (row in SELECT p.id AS id, u.displayName AS author, p.title AS title
					 FROM problems p
					 JOIN users u ON p.author == u.id) {
				Error error = findError(row.id, -1, -1);
				result << ProblemInfo(row.id, row.title, row.author, error.any(), false);
			}
		}
		result;
	}

	// Get an initial problem. i.e. exactly what is stored in the problems table. No user contributions.
	Problem initialProblem(Int problemId) {
		var result = WITH db: SELECT ONE title, impl, test, refimpl, checks, acceptableErrors
			FROM problems WHERE id == problemId;
		unless (result)
			throw ServerError("Problem ${problemId} does not exist.");

		var impl = getCode(result.impl);
		var test = getCode(result.test);
		var ref = getCode(result.refimpl);
		var error = findError(problemId, -1, -1);
		var checks = ProblemChecks:fromDB(result.checks);
		var accept = AcceptableErrors:fromDB(result.acceptableErrors);
		Problem(problemId, result.title, -1, 0, -1, 0, error, impl, test, ref, checks, accept);
	}

	// Get a problem.
	Problem problem(Int problemId, Int implId, Int testId) {
		WITH db {
			var problem = SELECT ONE title, impl, test, refimpl, checks, acceptableErrors
				FROM problems WHERE id == problemId;
			unless (problem)
				throw ServerError("Problem ${problemId} does not exist.");

			Int implCodeId = problem.impl;
			Int implVersion = 0;
			if (implId >= 0) {
				if (row = SELECT ONE code, version FROM implementations WHERE id == implId AND problem == problemId) {
					implCodeId = row.code;
					implVersion = row.version;
				}
			}

			Int testCodeId = problem.test;
			Int testVersion = 0;
			if (testId >= 0) {
				if (row = SELECT ONE code, version FROM tests WHERE id == testId AND problem == problemId) {
					testCodeId = row.code;
					testVersion = row.version;
				}
			}

			var impl = getCode(implCodeId);
			var test = getCode(testCodeId);
			var ref = getCode(problem.refimpl);
			var error = findError(problemId, implId, testId);
			var checks = ProblemChecks:fromDB(problem.checks);
			var accept = AcceptableErrors:fromDB(problem.acceptableErrors);
			Problem(problemId, problem.title,
					implId, implVersion,
					testId, testVersion,
					error,
					impl, test, ref,
					checks, accept);
		}
	}

	// Get an improved problem.
	Problem improvedProblem(Int userId, Int problemId, Int improvedId, ImprovementType type) {
		Int implId = -1;
		Int testId = -1;

		if (type == ImprovementType:test) {
			implId = bestImplementation(userId, problemId);
			testId = improvedId;
		} else {
			testId = bestTest(userId, problemId);
			implId = improvedId;
		}

		return problem(problemId, implId, testId);
	}

	// Add a new attempt.
	Int newAttempt(Int userId, Int implId, Int testId, Str sequence, Str? error) {
		WITH db: INSERT INTO attempts(author, impl, test, sequence, error, created)
			VALUES (userId, implId, testId, sequence, error, CURRENT DATETIME);
	}

	// Add a new implementation.
	Int newImplementation(Int userId, Int problemId, Code code, Int testId, Error error) {
		WITH db {
			Int ver = 1;
			if (r = SELECT ONE version FROM implementations WHERE problem == problemId AND author == userId ORDER BY version DESC)
				ver = r.version + 1;

			Int codeId = createCode(code);

			Int implId = INSERT INTO implementations(problem, version, author, code, basedOn, created)
				VALUES (problemId, ver, userId, codeId, testId, CURRENT DATETIME);

			// Also add to the autotable, so we don't have to check it ourselves.
			addError(problemId, implId, testId, error);

			return implId;
		}
	}

	// Add a new test.
	Int newTest(Int userId, Int problemId, Code code, Int implId, Error error) {
		WITH db {
			Int ver = 1;
			if (r = SELECT ONE version FROM tests WHERE problem == problem AND author == userId ORDER BY version DESC)
				ver = r.version + 1;

			Int codeId = createCode(code);

			Int testId = INSERT INTO tests(problem, version, author, code, basedOn, created)
				VALUES (problemId, ver, userId, codeId, implId, CURRENT DATETIME);

			// Also add to the autotable, so we don't have to check it ourselves.
			addError(problemId, implId, testId, error);

			return testId;
		}
	}

	// Get current state for a user and a given problem.
	CurrentStateResponse currentState(Int userId, Int problemId) {
		WITH db {
			unless (problem = SELECT ONE title, impl, test, refimpl, checks, acceptableErrors
					FROM problems WHERE id == problemId)
				throw ServerError("Problem ${problemId} does not exist!");

			Int implId = -1;
			Int implVersion = 0;
			Int implCode = problem.impl;
			if (result = SELECT ONE id, version, code FROM implementations
				WHERE problem == problemId AND author == userId
				ORDER BY version DESC) {
				implId = result.id;
				implVersion = result.version;
				implCode = result.code;
			}

			Int[] testIds;
			Code[] testCode;
			Error[] errors;

			testIds << -1;
			testCode << getCode(problem.test);
			errors << findError(problemId, implId, -1);

			for (row in SELECT id, version, code FROM tests
					 WHERE problem == problemId AND author == userId
					 ORDER BY version ASC) {
				testIds << row.id;
				testCode << getCode(row.code);
				errors << findError(problemId, implId, row.id);
			}

			var checks = ProblemChecks:fromDB(problem.checks);
			var accept = AcceptableErrors:fromDB(problem.acceptableErrors);
			CurrentStateResponse(problemId, problem.title,
								implId, implVersion, getCode(implCode),
								testIds, testCode, errors,
								getCode(problem.refimpl), checks, accept);
		}
	}

	Improvement[] testImprovements(Int userId, Int problemId) {
		WITH db {
			// Get information about errors at the same time, so figure out the user's current best
			// implementation.
			Int implId = bestImplementation(userId, problemId);

			var query = SELECT t.id AS id, u.id AS userId, t.version AS version, u.displayName AS author,
				ar.impl AS implId, ar.error AS error
				FROM tests t
				JOIN users u ON u.id == t.author
				LEFT JOIN autoresults ar ON ar.problem == t.problem AND ar.impl == implId AND ar.test == t.id
				WHERE t.problem == problemId
				ORDER BY t.author ASC, version ASC;

			// Note: We simply grab *all* tests here. They are all relevant.
			Improvement[] result;
			for (row in query) {
				Error error = if (row.implId.empty) {
					Error:unknown();
				} else if (e = row.error) {
					Error:error(e);
				} else {
					Error:success();
				};
				result << Improvement(row.id, row.version, row.author, error);
			}

			return result;
		}
	}

	CodeTriplet[] testMissing(Int userId, Int problemId, Int testId) {
		WITH db {
			var query = SELECT i.id AS id
				FROM implementations i
				LEFT JOIN autoresults ar ON ar.problem == i.problem AND ar.impl == i.id AND ar.test == testId
				WHERE i.problem = problemId AND i.author == userId AND ar.test IS NULL
				ORDER BY i.version DESC;

			CodeTriplet[] result;
			for (row in query) {
				result << CodeTriplet(problemId, row.id, testId);
			}

			// To be nice: check the original as well.
			if (findError(problemId, -1, testId).unknown())
				result << CodeTriplet(problemId, -1, testId);

			return result;
		}
	}

	Improvement[] implImprovements(Int userId, Int problemId) {
		WITH db {
			var query = SELECT t.id AS id, u.id AS userId, t.version AS version, u.displayName AS author
				FROM implementations t
				JOIN users u ON u.id == t.author
				WHERE t.problem == problemId
				ORDER BY t.author ASC, version DESC;

			// Note: Due to ordering, this is enough to only grab the highest version for each user.
			// They are arranged by author first, then by version. So we only need to detect when a
			// new user appears, and then grab the first row then.
			Int lastUser = -1;
			Improvement[] result;
			for (row in query) {
				if (row.userId == lastUser)
					continue;

				lastUser = row.userId;

				result << Improvement(row.id, row.version, row.author, Error:unknown());
			}

			// Find *any* of our tests that show an error. Note: It *might* be possible to do this
			// with additional JOINs in the query above, but it gets tricky quickly...
			for (r in result) {
				var query = SELECT t.version AS version, ar.test AS id, ar.error AS error
					FROM tests t
					LEFT JOIN autoresults ar ON ar.problem == t.problem AND ar.test == t.id AND ar.impl == ${r.id}
					WHERE t.problem == problemId AND t.author == userId
					ORDER BY t.version DESC;

				r.error = Error:success();
				for (row in query) {
					if (row.id.empty) {
						r.error = Error:unknown();
					} else if (rowError = row.error) {
						r.error = Error:error(rowError);
						r.errorVersion = row.version;
						break;
					}
				}
			}

			return result;
		}
	}

	CodeTriplet[] implMissing(Int userId, Int problemId, Int implId) {
		WITH db {
			var query = SELECT t.id AS id
				FROM tests t
				LEFT JOIN autoresults ar ON ar.problem == t.problem AND ar.test == t.id AND ar.impl == implId
				WHERE t.problem == problemId AND t.author == userId AND ar.test IS NULL
				ORDER BY t.version DESC;

			CodeTriplet[] result;
			for (row in query) {
				result << CodeTriplet(problemId, implId, row.id);
			}

			// To be nice: check the original as well.
			if (findError(problemId, implId, -1).unknown())
				result << CodeTriplet(problemId, implId, -1);

			return result;
		}
	}

	Code testCode(Int testId) {
		unless (row = WITH db: SELECT ONE code FROM tests WHERE id == testId)
			throw ServerError("No test with the id ${testId}.");

		return getCode(row.code);
	}

	Code implCode(Int implId) {
		unless (row = WITH db: SELECT ONE code FROM implementations WHERE id == implId)
			throw ServerError("No implementation with the id ${implId}.");

		return getCode(row.code);
	}

	// Add an error row. Returns "true" if we inserted the error.
	Bool addError(Int problemId, Int implId, Int testId, Error error) {
		try {
			if (error.unknown)
				return true;

			// Note: error.error returns NULL when it contains a success (we checked for unknown previously).
			WITH db: INSERT INTO autoresults(problem, impl, test, error)
				VALUES (problemId, implId, testId, ${error.error()});

			return true;
		} catch (SQLError error) {
			// Happens with duplicates. We ignore it here since we race with the auto-grader.
			return false;
		}
	}

	// Check if a problem has an error, and return the Bool? that is used in many parts of the system.
	Error findError(Int problemId, Int implId, Int testId) {
		unless (r = WITH db: SELECT ONE error FROM autoresults
				WHERE problem == problemId AND impl == implId AND test == testId)
			return Error:unknown();

		if (e = r.error)
			return Error:error(e);
		return Error:success();
	}

	// Get all versions of implementations.
	Int[] allImplementations(Int problemId) {
		WITH db {
			Int[] result;

			for (row in SELECT id, author FROM implementations
					 WHERE problem == problemId
					 ORDER BY author ASC, version DESC) {
				result << row.id;
			}

			return result;
		}
	}

	// Get all versions of tests.
	Int[] allTests(Int problemId) {
		WITH db {
			Int[] result;

			for (row in SELECT id, author FROM tests
					 WHERE problem == problemId
					 ORDER BY author ASC, version DESC) {

				result << row.id;
			}

			return result;
		}
	}

	/**
	 * Utils
	 */

	private Int createCode(Code code) {
		WITH db: INSERT INTO code(src, language) VALUES (${code.src}, ${code.language});
	}

	private Code getCode(Int codeId) {
		var code = WITH db: SELECT ONE src, language FROM code WHERE id == codeId;
		unless (code)
			throw ServerError("Code ${codeId} does not exist.");

		Code(codeId, code.src, code.language);
	}

	// Validate implementation ID.
	private void checkImplId(Int implId) {
		unless (WITH db: SELECT ONE id FROM implementations WHERE id == implId)
			throw ServerError("No implementation with id ${implId}.");
	}

	// Validate test ID.
	private void checkTestId(Int testId) {
		unless (WITH db: SELECT ONE id FROM tests WHERE id == testId)
			throw ServerError("No test with id ${testId}.");
	}

	// Compare code for equality to disallow submitting multiple instances of essentially the same
	// solution to hoard points. Throws on failure.
	private void compareCode(Int original, Code newCode) {
		unless (code = WITH db: SELECT ONE src FROM code WHERE id == original)
			return;

		if (codeSignature(code.src) == newCode.signature)
			throw ServerError("This solution is almost identical to another solution you submitted to this problem.");
	}

	// Get the user's current best implementation for a problem.
	private Int bestImplementation(Int userId, Int problemId) {
		if (r = WITH db: SELECT ONE id FROM implementations
			WHERE author == userId AND problem == problemId
			ORDER BY version DESC) {
			return r.id;
		} else {
			return -1;
		}
	}

	// Get the user's current best test for a problem.
	private Int bestTest(Int userId, Int problemId) {
		if (r = WITH db: SELECT ONE id FROM tests
			WHERE author == userId AND problem == problemId
			ORDER BY version DESC) {
			return r.id;
		} else {
			return -1;
		}
	}
}

class UserInfo {
	// User ID.
	Int id;

	// Display name of the user.
	Str name;

	// Is the user an admin?
	Bool admin;

	init(Int id, Str name, Bool admin) {
		init { id = id; name = name; admin = admin; }
	}
}
