/*
 * Copyright (C) 2025  Isaac Joseph <calamityjoe87@gmail.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 */

using GLib;
using Gee;

/**
 * FeedUpdateManager - Manages automatic updates of RSS feeds
 * Updates all followed RSS feeds on app startup and tracks update timestamps
 */

public class FeedUpdateManager : GLib.Object {
    private weak NewsWindow window;
    private HashMap<string, int64?> update_timestamps;
    private bool is_updating = false;
    private uint recurring_timer_id = 0;

    // Signals for UI operations
    public signal void request_show_toast(string message);

    public FeedUpdateManager(NewsWindow window) {
        this.window = window;
        this.update_timestamps = new HashMap<string, int64?>();
    }

    ~FeedUpdateManager() {
        stop_recurring_updates();
    }

    /**
     * Convert user's update_interval preference to seconds
     * @return Number of seconds, or -1 if manual updates
     */
    private int64 get_update_interval_seconds() {
        string interval = window.prefs.update_interval;

        switch (interval) {
            case "manual":
                return -1; // No automatic updates
            case "15min":
                return 900;  // 15 * 60
            case "30min":
                return 1800; // 30 * 60
            case "1hour":
                return 3600; // 60 * 60
            case "2hours":
                return 7200; // 2 * 60 * 60
            case "4hours":
                return 14400; // 4 * 60 * 60
            default:
                return 3600; // Default to 1 hour
        }
    }

    /**
     * Start recurring feed updates based on user's interval preference
     * Always performs an immediate update on app launch (after delay)
     */
    public void start_recurring_updates() {
        // Check if automatic updates are disabled
        int64 update_interval = get_update_interval_seconds();
        if (update_interval == -1) {
            GLib.print("Automatic updates disabled (manual mode)\n");
            return;
        }

        // Always update on app launch (users expect fresh content)
        GLib.print("App launched - updating all feeds (interval: %lld seconds)\n", update_interval);
        update_all_feeds_async();

        // Start recurring timer to update while app is running
        stop_recurring_updates(); // Clear any existing timer

        recurring_timer_id = GLib.Timeout.add_seconds((uint)update_interval, () => {
            GLib.print("Recurring update timer fired (interval: %lld seconds)\n", update_interval);
            update_all_feeds_async();
            return GLib.Source.CONTINUE; // Keep running
        });

        GLib.print("Started recurring update timer (interval: %lld seconds)\n", update_interval);
    }

    /**
     * Stop the recurring update timer
     */
    public void stop_recurring_updates() {
        if (recurring_timer_id > 0) {
            GLib.Source.remove(recurring_timer_id);
            recurring_timer_id = 0;
            GLib.print("Stopped recurring update timer\n");
        }
    }

    /**
     * Update all RSS feeds asynchronously
     *
     * This method runs in a background thread and updates all feeds
     * that haven't been updated within the user's configured interval
     */
    public void update_all_feeds_async() {
        if (is_updating) {
            GLib.print("Feed update already in progress, skipping\n");
            return;
        }

        // Check if automatic updates are disabled
        int64 update_interval = get_update_interval_seconds();
        if (update_interval == -1) {
            GLib.print("Automatic updates disabled (manual mode)\n");
            return;
        }

        is_updating = true;

        new Thread<void*>("feed-updater", () => {
            var store = Paperboy.RssSourceStore.get_instance();
            var sources = store.get_all_sources();

            if (sources.size == 0) {
                GLib.print("No RSS feeds to update\n");
                is_updating = false;
                return null;
            }

            GLib.print("Checking %d RSS feeds for updates (interval: %lld seconds)...\n",
                      sources.size, update_interval);

            int updated_count = 0;
            int skipped_count = 0;
            int failed_count = 0;

            foreach (var source in sources) {
                // Check if feed needs update (based on last_fetched_at)
                int64 now = GLib.get_real_time() / 1000000;
                int64 time_since_fetch = now - source.last_fetched_at;

                if (time_since_fetch < update_interval) {
                    GLib.print("  ⏭  Skipping %s (updated %lld seconds ago)\n",
                        source.name, time_since_fetch);
                    skipped_count++;
                    continue;
                }
                
                // Fetch and validate feed
                bool success = update_single_feed(source);
                if (success) {
                    updated_count++;
                } else {
                    failed_count++;
                }
                
                // Small delay between requests to avoid overwhelming servers
                Thread.usleep(500000); // 500ms
            }
            
            // Show summary toast
            GLib.Idle.add(() => {
                if (window == null) return false; // Window destroyed
                if (updated_count > 0 || failed_count > 0) {
                    string message = "RSS feeds: %d updated".printf(updated_count);
                    if (failed_count > 0) {
                        message += ", %d failed".printf(failed_count);
                    }
                    request_show_toast(message);
                }
                return false;
            });
            
            GLib.print("Feed update complete: %d updated, %d skipped, %d failed\n", 
                updated_count, skipped_count, failed_count);
            
            is_updating = false;
            return null;
        });
    }
    
    /**
     * Update a single RSS feed
     * 
     * @param source The RSS source to update
     * @return true if successful, false otherwise
     */
    private bool update_single_feed(Paperboy.RssSource source) {
        // Handle file:// URLs (locally generated feeds) - REGENERATE instead of just validate
        if (source.url.has_prefix("file://")) {
            // Check if we have the original_url to regenerate from
            if (source.original_url == null || source.original_url.length == 0) {
                GLib.warning("  ✗ Cannot regenerate %s: no original_url stored", source.name);
                return false;
            }

            GLib.print("  ⟳ Regenerating feed for %s from %s\n", source.name, source.original_url);

            // Extract host from original_url
            string? host = UrlUtils.extract_host_from_url(source.original_url);
            if (host == null || host.length == 0) {
                GLib.warning("  ✗ Cannot extract host from original_url: %s", source.original_url);
                return false;
            }

            // Find html2rss binary
            string? html2rss_path = find_html2rss_binary();
            if (html2rss_path == null) {
                GLib.warning("  ✗ html2rss binary not found");
                return false;
            }

            try {
                // Run html2rss to generate fresh feed
                // Match the argument format from sourceManager.vala
                string[] argv = {html2rss_path, "--max-pages", "20", source.original_url};
                var proc = new GLib.Subprocess.newv(argv, GLib.SubprocessFlags.STDOUT_PIPE | GLib.SubprocessFlags.STDERR_PIPE);
                
                string? stdout_str = null;
                string? stderr_str = null;
                proc.communicate_utf8(null, null, out stdout_str, out stderr_str);
                proc.wait();
                
                int exit_status = proc.get_exit_status();
                string gen_feed = stdout_str != null ? stdout_str : "";

                if (exit_status == 0 && gen_feed.length > 0) {
                    // Validate the generated feed
                    string? error = null;
                    if (RssValidatorUtils.is_valid_rss(gen_feed, out error)) {
                        int item_count = RssValidatorUtils.get_item_count(gen_feed);
                        
                        // Check if content actually changed by comparing with old feed
                        bool content_changed = true;
                        string old_file_path = "";
                        if (source.url.length > 7) {
                            old_file_path = source.url.substring(7); // Remove "file://" prefix
                        }

                        if (old_file_path.length > 0) {
                            try {
                                var old_file = GLib.File.new_for_path(old_file_path);
                                if (old_file.query_exists()) {
                                // Read old feed content
                                uint8[] old_contents;
                                old_file.load_contents(null, out old_contents, null);
                                string old_feed = (string) old_contents;
                                
                                // Compare feeds by checking if they have the same items
                                // We'll use a simple heuristic: compare item count and a hash of GUIDs/links
                                if (RssValidatorUtils.is_valid_rss(old_feed, out error)) {
                                    int old_item_count = RssValidatorUtils.get_item_count(old_feed);
                                    
                                    if (old_item_count == item_count) {
                                        // Same number of items - do a deeper comparison
                                        // Extract GUIDs/links from both feeds and compare
                                        string old_signature = extract_feed_signature(old_feed);
                                        string new_signature = extract_feed_signature(gen_feed);
                                        
                                        if (old_signature == new_signature) {
                                            content_changed = false;
                                            GLib.print("  ⏭  Skipping %s - content unchanged (%d items)\n", source.name, item_count);
                                        }
                                    }
                                }
                                }
                            } catch (Error e) {
                            // If we can't read old file, assume content changed
                            GLib.warning("  ⚠ Could not read old feed for comparison: %s", e.message);
                        }
                        }
                        
                        // Only update if content actually changed
                        if (content_changed) {
                            // Merge old articles into new feed before saving
                            if (old_file_path.length > 0) {
                                try {
                                    var old_file = GLib.File.new_for_path(old_file_path);
                                    if (old_file.query_exists()) {
                                        // Read old feed content
                                        uint8[] old_contents;
                                        old_file.load_contents(null, out old_contents, null);
                                        string old_feed = (string) old_contents;

                                        if (RssValidatorUtils.is_valid_rss(old_feed, out error)) {
                                            // Merge old articles into new feed
                                            gen_feed = merge_rss_feeds(old_feed, gen_feed);
                                            item_count = RssValidatorUtils.get_item_count(gen_feed);
                                            GLib.print("  ↻ Merged with old feed (now %d total items)\n", item_count);
                                        }
                                    }
                                } catch (Error e) {
                                    GLib.warning("  ⚠ Failed to merge old feed: %s", e.message);
                                }
                            }

                            // Delete old XML file (will be replaced by merged version)
                            if (old_file_path.length > 0) {
                                try {
                                    var old_file = GLib.File.new_for_path(old_file_path);
                                    if (old_file.query_exists()) {
                                        old_file.delete();
                                        GLib.print("  ✓ Deleted old feed file: %s\n", GLib.Path.get_basename(old_file_path));
                                    }
                                } catch (Error e) {
                                    GLib.warning("  ⚠ Failed to delete old feed file: %s", e.message);
                                }
                            }

                            // Save new XML file (now contains merged articles)
                            // Use same filename (without timestamp) so we replace the old file
                            string data_dir = GLib.Environment.get_user_data_dir();
                            string paperboy_dir = GLib.Path.build_filename(data_dir, "paperboy");
                            string gen_dir = GLib.Path.build_filename(paperboy_dir, "generated_feeds");
                            GLib.DirUtils.create_with_parents(gen_dir, 0755);

                            string safe_host = host.replace("/", "_").replace(":", "_");
                            string filename = safe_host + ".xml";
                            string new_file_path = GLib.Path.build_filename(gen_dir, filename);

                            var f = GLib.File.new_for_path(new_file_path);
                            var out_stream = f.replace(null, false, GLib.FileCreateFlags.NONE, null);
                            var writer = new DataOutputStream(out_stream);
                            string safe_feed = RssValidatorUtils.sanitize_for_xml(gen_feed);
                            writer.put_string(safe_feed);
                            writer.close(null);

                            // Update database with new file path
                            var store = Paperboy.RssSourceStore.get_instance();
                            string new_url = "file://" + new_file_path;
                            store.update_source_url(source.url, new_url);
                            store.update_last_fetched(new_url);

                            GLib.print("  ✓ Regenerated: %s (%d items)\n", source.name, item_count);
                        } else {
                            // Content unchanged, just update timestamp
                            var store = Paperboy.RssSourceStore.get_instance();
                            store.update_last_fetched(source.url);
                        }
                        
                        return true;
                    } else {
                        GLib.warning("  ✗ Generated invalid RSS for %s: %s", source.name, error);
                        return false;
                    }
                } else {
                    GLib.warning("  ✗ html2rss failed for %s (exit=%d)", source.name, exit_status);
                    if (stderr_str != null && stderr_str.length > 0) {
                        GLib.warning("  ✗ html2rss stderr: %s", stderr_str);
                    }
                    return false;
                }
            } catch (Error e) {
                GLib.warning("  ✗ Error regenerating feed for %s: %s", source.name, e.message);
                return false;
            }
        }
        
        // Handle HTTP/HTTPS URLs
        try {
            var msg = new Soup.Message("GET", source.url);
            msg.get_request_headers().append("User-Agent", "paperboy/0.5.1a");
            msg.get_request_headers().append("Accept", "application/rss+xml, application/atom+xml, application/xml, text/xml");
            
            GLib.Bytes? response = window.session.send_and_read(msg, null);
            var status = msg.get_status();
            
            if (status == Soup.Status.OK && response != null) {
                string body = (string) response.get_data();
                
                // Validate RSS
                string? error = null;
                if (RssValidatorUtils.is_valid_rss(body, out error)) {
                    int item_count = RssValidatorUtils.get_item_count(body);
                    
                    // Update last_fetched_at timestamp
                    var store = Paperboy.RssSourceStore.get_instance();
                    store.update_last_fetched(source.url);
                    
                    GLib.print("  ✓ Updated: %s (%d items)\n", source.name, item_count);
                    return true;
                } else {
                    GLib.warning("  ✗ Invalid RSS for %s: %s", source.name, error);
                    return false;
                }
            } else {
                GLib.warning("  ✗ Failed to fetch %s: HTTP %u", source.name, status);
                return false;
            }
        } catch (Error e) {
            GLib.warning("  ✗ Error updating %s: %s", source.name, e.message);
            return false;
        }
    }
    
    /**
     * Extract a signature from an RSS feed based on item GUIDs/links
     * This is used to determine if feed content has changed
     * @param feed_xml The RSS feed XML content
     * @return A signature string representing the feed's items
     */
    private string extract_feed_signature(string feed_xml) {
        var signature = new StringBuilder();
        
        try {
            Xml.Doc* doc = Xml.Parser.parse_doc(feed_xml);
            if (doc == null) {
                return "";
            }
            
            Xml.Node* root = doc->get_root_element();
            if (root == null) {
                delete doc;
                return "";
            }
            
            // Find all <item> or <entry> elements
            for (Xml.Node* node = root->children; node != null; node = node->next) {
                if (node->type != Xml.ElementType.ELEMENT_NODE) continue;
                
                // Handle RSS <channel> wrapper
                if (node->name == "channel") {
                    for (Xml.Node* item = node->children; item != null; item = item->next) {
                        if (item->type != Xml.ElementType.ELEMENT_NODE) continue;
                        if (item->name == "item") {
                            extract_item_signature(item, signature);
                        }
                    }
                }
                // Handle Atom <entry> elements directly under root
                else if (node->name == "entry") {
                    extract_item_signature(node, signature);
                }
            }
            
            delete doc;
        } catch (Error e) {
            GLib.warning("Error extracting feed signature: %s", e.message);
        }
        
        return signature.str;
    }
    
    /**
     * Extract signature from a single RSS item or Atom entry
     */
    private void extract_item_signature(Xml.Node* item, StringBuilder signature) {
        for (Xml.Node* child = item->children; child != null; child = child->next) {
            if (child->type != Xml.ElementType.ELEMENT_NODE) continue;
            
            // Look for GUID, link, or id elements
            if (child->name == "guid" || child->name == "link" || child->name == "id") {
                string? content = child->get_content();
                if (content != null && content.length > 0) {
                    signature.append(content);
                    signature.append("|");
                }
            }
        }
    }
    
    /**
     * Merge two RSS feeds, combining items from both while avoiding duplicates
     * @param old_feed The old RSS feed XML
     * @param new_feed The new RSS feed XML
     * @return Merged RSS feed XML with items from both feeds
     */
    private string merge_rss_feeds(string old_feed, string new_feed) {
        try {
            // Parse both feeds
            Xml.Doc* old_doc = Xml.Parser.parse_doc(old_feed);
            Xml.Doc* new_doc = Xml.Parser.parse_doc(new_feed);

            if (old_doc == null || new_doc == null) {
                GLib.warning("Failed to parse feeds for merging");
                if (old_doc != null) delete old_doc;
                if (new_doc != null) delete new_doc;
                return new_feed; // Return new feed as fallback
            }

            Xml.Node* old_root = old_doc->get_root_element();
            Xml.Node* new_root = new_doc->get_root_element();

            if (old_root == null || new_root == null) {
                delete old_doc;
                delete new_doc;
                return new_feed;
            }

            // Find the channel node in new feed (RSS) or use root for Atom
            Xml.Node* new_channel = null;
            for (Xml.Node* node = new_root->children; node != null; node = node->next) {
                if (node->type == Xml.ElementType.ELEMENT_NODE && node->name == "channel") {
                    new_channel = node;
                    break;
                }
            }

            // If no channel found, assume Atom feed or malformed RSS
            if (new_channel == null) {
                new_channel = new_root;
            }

            // Collect new item GUIDs/links to avoid duplicates
            var new_item_ids = new Gee.HashSet<string>();
            for (Xml.Node* item = new_channel->children; item != null; item = item->next) {
                if (item->type == Xml.ElementType.ELEMENT_NODE && item->name == "item") {
                    string? id = extract_item_id(item);
                    if (id != null) {
                        new_item_ids.add(id);
                    }
                }
            }

            // Find old channel
            Xml.Node* old_channel = null;
            for (Xml.Node* node = old_root->children; node != null; node = node->next) {
                if (node->type == Xml.ElementType.ELEMENT_NODE && node->name == "channel") {
                    old_channel = node;
                    break;
                }
            }

            if (old_channel == null) {
                old_channel = old_root;
            }

            // Copy old items that aren't in new feed
            int merged_count = 0;
            for (Xml.Node* item = old_channel->children; item != null; item = item->next) {
                if (item->type == Xml.ElementType.ELEMENT_NODE && item->name == "item") {
                    string? id = extract_item_id(item);
                    if (id != null && !new_item_ids.contains(id)) {
                        // Copy this item to new feed
                        Xml.Node* copied_item = item->copy(1); // Deep copy
                        new_channel->add_child(copied_item);
                        merged_count++;
                    }
                }
            }

            // Convert back to string
            string result = "";
            new_doc->dump_memory_enc(out result);

            delete old_doc;
            delete new_doc;

            GLib.print("  ✓ Merged %d unique old articles into new feed\n", merged_count);
            return result;

        } catch (Error e) {
            GLib.warning("Error merging feeds: %s", e.message);
            return new_feed; // Return new feed as fallback
        }
    }

    /**
     * Extract a unique identifier from an RSS item (GUID or link)
     * @param item The RSS item node
     * @return The item's unique identifier or null
     */
    private string? extract_item_id(Xml.Node* item) {
        for (Xml.Node* child = item->children; child != null; child = child->next) {
            if (child->type != Xml.ElementType.ELEMENT_NODE) continue;

            if (child->name == "guid" || child->name == "link") {
                string? content = child->get_content();
                if (content != null && content.length > 0) {
                    return content.strip();
                }
            }
        }
        return null;
    }

    /**
     * Find the html2rss binary in various possible locations
     * @return Path to html2rss binary or null if not found
     */
    private string? find_html2rss_binary() {
        // Fallback candidates: system/AppImage locations and developer build locations
        var candidates = new Gee.ArrayList<string>();

        // Environment variable override (AppRun / AppImage)
        string? env_libexec = GLib.Environment.get_variable("PAPERBOY_LIBEXECDIR");
        if (env_libexec != null && env_libexec.length > 0) {
            candidates.add(GLib.Path.build_filename(env_libexec, "paperboy", "html2rss"));
            candidates.add(GLib.Path.build_filename(env_libexec, "html2rss"));
        }

        // Build-time libexecdir (system install)
        string libexec = BuildConstants.LIBEXECDIR;
        if (libexec != null && libexec.length > 0) {
            candidates.add(GLib.Path.build_filename(libexec, "paperboy", "html2rss"));
        }

        // FHS-compliant: check libexecdir for internal binaries
        // Standard system locations (prefix-aware)
        candidates.add("/usr/libexec/paperboy/html2rss");
        candidates.add("/usr/local/libexec/paperboy/html2rss");

        // Flatpak/AppImage locations
        candidates.add("/app/libexec/paperboy/html2rss");

        // Development build locations (for running from source tree)
        candidates.add("tools/html2rss/target/release/html2rss");
        candidates.add("./tools/html2rss/target/release/html2rss");
        candidates.add("../tools/html2rss/target/release/html2rss");

        string? cwd = GLib.Environment.get_current_dir();
        if (cwd != null) {
            candidates.add(GLib.Path.build_filename(cwd, "tools", "html2rss", "target", "release", "html2rss"));
        }

        // Check each candidate and return the first executable match
        foreach (string c in candidates) {
            try {
                if (GLib.FileUtils.test(c, GLib.FileTest.EXISTS) && GLib.FileUtils.test(c, GLib.FileTest.IS_EXECUTABLE)) {
                GLib.message("Using html2rss at: %s", c);
                return c;
                }
            } catch (GLib.Error e) {
                // Not found
                AppDebugger.log_if_enabled("/tmp/paperboy-debug.log", "html2rss not found in candidates");
            }
        }

        return null;
    }
    
    /**
     * Force update a specific feed by URL
     * 
     * @param feed_url The URL of the feed to update
     * @return true if successful, false otherwise
     */
    public bool force_update_feed(string feed_url) {
        var store = Paperboy.RssSourceStore.get_instance();
        var source = store.get_source_by_url(feed_url);
        
        if (source == null) {
            GLib.warning("Feed not found: %s", feed_url);
            return false;
        }
        
        return update_single_feed(source);
    }
}
