/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright(C) 2009,...,2018 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */

#pragma once

/////////////////////// Stdlib includes
#include <vector>


/////////////////////// Qt includes
#include <QString>
#include <QList>


/////////////////////// Local includes
#include "MsXpS/libXpertMassCore/globals.hpp"
#include "MsXpS/export-import-config.h"
#include "MsXpS/libXpertMassCore/PolChemDef.hpp"
#include "MsXpS/libXpertMassCore/FragmentationRule.hpp"


namespace MsXpS
{
namespace libXpertMassCore
{


class FragmentationPathway;
typedef std::shared_ptr<FragmentationPathway> FragmentationPathwaySPtr;
typedef std::shared_ptr<const FragmentationPathway> FragmentationPathwayCstSPtr;


class DECLSPEC FragmentationPathway
{
  public:
  FragmentationPathway(PolChemDefCstSPtr pol_chem_def_csp,
                       const QDomElement &element,
                       int version);
  FragmentationPathway(PolChemDefCstSPtr pol_chem_def_csp = nullptr,
                       const QString &name                = QString(),
                       const QString &formula_string      = QString(),
                       Enums::FragEnd frag_end                   = Enums::FragEnd::NE,
                       const QString &comment             = QString());
  FragmentationPathway(const FragmentationPathway &other);
  virtual ~FragmentationPathway();

  //////////////// THE POLCHEMDEF /////////////////////
  void setPolchemDefCstSPtr(PolChemDefCstSPtr pol_chem_def_csp);
  PolChemDefCstSPtr getPolchemDefCstSPtr() const;

  //////////////// THE NAME /////////////////////
  void setName(const QString &name);
  const QString &getName() const;

  //////////////// THE FORMULA /////////////////////
  void setFormula(const Formula &formula);
  void setFormula(const QString &formula_string);
  const Formula &getFormulaCstRef() const;
  Formula &getFormulaRef();

  //////////////// THE FRAG END /////////////////////
  void setFragEnd(Enums::FragEnd frag_end);
  Enums::FragEnd getFragEnd() const;

  //////////////// THE MONOMER CONTRIBUTION /////////////////////
  void setMonomerContribution(int monomer_contribution);
  int getMonomerContribution() const;

  //////////////// THE COMMENT /////////////////////
  void setComment(const QString &comment);
  QString getComment() const;

  //////////////// THE RULES CONTAINER /////////////////////
  const std::vector<FragmentationRuleSPtr> &getRulesCstRef() const;
  std::vector<FragmentationRuleSPtr> &getRulesRef();

  //////////////// OPERATORS /////////////////////
  FragmentationPathway &operator=(const FragmentationPathway &other);
  bool operator==(const FragmentationPathway &other) const;
  bool operator!=(const FragmentationPathway &other) const;

  //////////////// RULES /////////////////////
  void addRule(FragmentationRuleSPtr frag_rule_sp);
  void insertRuleAt(FragmentationRuleSPtr frag_rule_sp, std::size_t index);
  void removeRuleAt(size_t index);

  //////////////// VALIDATIONS /////////////////////
  FragmentationPathwayCstSPtr getFromPolChemDefByName() const;
  Enums::PolChemDefEntityStatus isKnownByNameInPolChemDef() const;

  bool validate(ErrorList *error_list_p) const;
  bool isValid() const;

  //////////////// UTILITIES /////////////////////
  QString toString() const;

  //////////////// XML DATA LOADING WRITING /////////////////////
  bool renderXmlFgsElement(const QDomElement &element, int version = 1);
  bool renderXmlFgpElement(const QDomElement &element, int version = 2);
  QString
  formatXmlFgpElement(int offset,
                      const QString &indent = Utils::xmlIndentationToken) const;

  protected:
  PolChemDefCstSPtr mcsp_polChemDef = nullptr;
  QString m_name                    = "";
  Formula m_formula;
  Enums::FragEnd m_fragEnd = Enums::FragEnd::NE;
  //! Fragmented monomer's mass contribution. See fragSpecDefDlg.cpp
  //! for a detailed explanation of what this member is for.
  int m_monomerContribution = 0;
  QString m_comment;
  std::vector<FragmentationRuleSPtr> m_rules;
  mutable bool m_isValid = false;
};


} // namespace libXpertMassCore
} // namespace MsXpS
