/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

#ifndef FIELD_H
#define FIELD_H

#include <cstdio>
#include "array.h"
#include "compare.h"

double varToStd(double rvar, double missval);

enum field_flag
{
  FIELD_NONE = 1,
  FIELD_VEC = 2,
  FIELD_FLT = 4,
};

class Field
{
public:
  int fpeRaised;
  int nwpv;            // number of words per value; real:1  complex:2
  int floatPrecision;  // float or double
  int grid;

  size_t size;
  size_t nsamp;

  size_t nmiss;
  double missval;

  Varray<float> vecf;
  Varray<double> vec;
  Varray<double> weightv;

  Field();
  void resize(size_t count);
  void resize(size_t count, double value);
  void resizef(size_t count);
  void resizef(size_t count, float value);
  bool empty() const;
  void check_gridsize() const;

private:
  size_t m_count;
};

struct RecordInfo
{
  int varID;
  int levelID;
  bool lconst;
};

using FieldVector = std::vector<Field>;
using FieldVector2D = std::vector<FieldVector>;
using FieldVector3D = std::vector<FieldVector2D>;

void fieldFill(Field &field, double value);
size_t fieldNumMiss(const Field &field);

// fieldmem.cc
void fieldsFromVlist(int vlistID, FieldVector2D &field2D, int ptype);
void fieldsFromVlist(int vlistID, FieldVector2D &field2D, int ptype, double fillValue);

// field.cc
double vfldfun(const Field &field, int function);
double vfldmin(const Field &field);
double vfldmax(const Field &field);
double vfldsum(const Field &field);
double vfldmean(const Field &field);
double vfldmeanw(const Field &field);
double vfldavg(const Field &field);
double vfldavgw(const Field &field);
double vfldstd(const Field &field);
double vfldstd1(const Field &field);
double vfldvar(const Field &field);
double vfldvar1(const Field &field);
double vfldstdw(const Field &field);
double vfldstd1w(const Field &field);
double vfldvarw(const Field &field);
double vfldvar1w(const Field &field);

// ENS VALIDATION
double vfldrank(Field &field);

double vfldpctl(Field &field, double pn);

// fieldzon.cc
void zonfun(const Field &field1, Field &field2, int function);
void zonmin(const Field &field1, Field &field2);
void zonmax(const Field &field1, Field &field2);
void zonrange(const Field &field1, Field &field2);
void zonsum(const Field &field1, Field &field2);
void zonavg(const Field &field1, Field &field2);
void zonmean(const Field &field1, Field &field2);
void zonstd(const Field &field1, Field &field2);
void zonstd1(const Field &field1, Field &field2);
void zonvar(const Field &field1, Field &field2);
void zonvar1(const Field &field1, Field &field2);
void zonpctl(Field &field1, Field &field2, int p);

// fieldmer.cc
void merfun(const Field &field1, Field &field2, int function);
void merpctl(Field &field1, Field &field2, int p);

void vfldrms(const Field &field1, const Field &field2, Field &field3);

// fieldc.cc
void vfarcfun(Field &field, double rconst, int function);

void vfarcmul(Field &field, double rconst);
void vfarcdiv(Field &field, double rconst);
void vfarcadd(Field &field, double rconst);
void vfarcsub(Field &field, double rconst);
void vfarcmin(Field &field, double rconst);
void vfarcmax(Field &field, double rconst);
void vfarmod(Field &field, double divisor);

// fieldccplx.cc
void vfarcfuncplx(Field &field, const double rconstcplx[2], int function);

// field2.cc
void vfarfun(Field &field1, const Field &field2, int function);

void vfaradd(Field &field1, const Field &field2);
void vfarsum(Field &field1, const Field &field2);
void vfarsub(Field &field1, const Field &field2);
void vfarmul(Field &field1, const Field &field2);
void vfardiv(Field &field1, const Field &field2);
void vfarmin(Field &field1, const Field &field2);
void vfarmax(Field &field1, const Field &field2);
void vfaratan2(Field &field1, const Field &field2);

void vfarsumq(Field &field1, const Field &field2);
void vfarsumw(Field &field1, const Field &field2, double w);
void vfarsumqw(Field &field1, const Field &field2, double w);
void vfarsumtr(Field &field1, const Field &field2, double refval);
void vfarcpy(Field &field1, const Field &field2);

void vfarminidx(Field &field1, Field &field2, const Field &field3, int idx);
void vfarmaxidx(Field &field1, Field &field2, const Field &field3, int idx);
void vfarvar(Field &field1, const Field &field2, const Field &field3, int divisor);
void vfarstd(Field &field1, const Field &field2, const Field &field3, int divisor);
void vfarcvar(Field &field1, const Field &field2, int nsets, int divisor);
void vfarcstd(Field &field1, const Field &field2, int nsets, int divisor);
void vfarmoq(Field &field1, const Field &field2);
void vfarmoqw(Field &field1, const Field &field2, double w);

void vfarcount(Field &field1, const Field &field2);

// field2cplx.cc
void vfarfuncplx(Field &field1, const Field &field2, int function);

#endif /* FIELD_H */
