#!/usr/bin/env bash

set -euxo pipefail

#
# Azure Pipelines does not allow an empty string as the value for a pipeline parameter; instead we use "-" to indicate
# an empty value. Change "-" to "" for the variables that capture the parameter values.
if [[ $TEST_SUITES == "-" ]]; then
    TEST_SUITES=""  # Don't set the test_suites variable
else
    TEST_SUITES="-v test_suites:\"$TEST_SUITES\""
fi
if [[ $TEST_ARGS == "-" ]]; then
    TEST_ARGS=""
fi
if [[ $IMAGE == "-" ]]; then
    IMAGE=""
fi
if [[ $LOCATIONS == "-" ]]; then
    LOCATIONS=""
fi
if [[ $VM_SIZE == "-" ]]; then
    VM_SIZE=""
fi
if [[ $CONTAINER_IMAGE == "-" ]]; then
    CONTAINER_IMAGE="waagenttests.azurecr.io/waagenttests:latest"
fi

echo "Hostname: $(hostname)"
echo "\$USER: $USER"

#
# UID of 'waagent' in the Docker container
#
WAAGENT_UID=1000

#
# Set the correct mode and owner for the private SSH key and generate the public key.
#
cd "$AGENT_TEMPDIRECTORY"
mkdir ssh
cp "$DOWNLOADSSHKEY_SECUREFILEPATH" ssh
chmod 700 ssh/id_rsa
ssh-keygen -y -f ssh/id_rsa > ssh/id_rsa.pub
sudo find ssh -exec chown "$WAAGENT_UID" {} \;

#
# Allow write access to the sources directory. This is needed because building the agent package (within the Docker
# container) writes the egg info to that directory.
#
chmod a+w "$BUILD_SOURCESDIRECTORY"

#
# Create the directory where the Docker container will create the test logs and give ownership to 'waagent'
#
LOGS_DIRECTORY="$AGENT_TEMPDIRECTORY/logs"
echo "##vso[task.setvariable variable=logs_directory]$LOGS_DIRECTORY"
mkdir "$LOGS_DIRECTORY"
sudo chown "$WAAGENT_UID" "$LOGS_DIRECTORY"

#
# Give the current user access to the Docker daemon
#
sudo usermod -aG docker $USER
newgrp docker < /dev/null

#
# Pull the container image used to execute the tests
#
az acr login --name waagenttests --username "$CR_USER" --password "$CR_SECRET"

docker pull "$CONTAINER_IMAGE"

#
# Get the external IP address of the VM.
#
IP_ADDRESS=$(curl -4 https://ifconfig.io/ip)
if [[ ! "${IP_ADDRESS}" =~ ^[0-9]+\.[0-9]+\.[0-9]+\.[0-9]+$ ]]; then
  echo "Failed to get the external IP address of the local machine. 'curl -4 https://ifconfig.io/ip' returned $IP_ADDRESS"
  exit 1
fi

# certificate location in the container
AZURE_CLIENT_CERTIFICATE_PATH="/home/waagent/app/cert.pem"
# Need to set this to True if we sue SNI based authentication for certificate
AZURE_CLIENT_SEND_CERTIFICATE_CHAIN="True"

docker run --rm \
    --volume "$BUILD_SOURCESDIRECTORY:/home/waagent/WALinuxAgent" \
    --volume "$AGENT_TEMPDIRECTORY"/ssh:/home/waagent/.ssh \
    --volume "$AGENT_TEMPDIRECTORY"/app:/home/waagent/app \
    --volume "$LOGS_DIRECTORY":/home/waagent/logs \
    --env AZURE_CLIENT_ID \
    --env AZURE_TENANT_ID \
    --env AZURE_CLIENT_CERTIFICATE_PATH=$AZURE_CLIENT_CERTIFICATE_PATH \
    --env AZURE_CLIENT_SEND_CERTIFICATE_CHAIN=$AZURE_CLIENT_SEND_CERTIFICATE_CHAIN \
    "$CONTAINER_IMAGE" \
    bash --login -c \
      "lisa \
          --runbook \$HOME/WALinuxAgent/tests_e2e/orchestrator/runbook.yml \
          --log_path \$HOME/logs/lisa \
          --working_path \$HOME/tmp \
          -v cloud:$CLOUD \
          -v subscription_id:$SUBSCRIPTION_ID \
          -v identity_file:\$HOME/.ssh/id_rsa \
          -v log_path:\$HOME/logs \
          -v collect_logs:\"$COLLECT_LOGS\" \
          -v keep_environment:\"$KEEP_ENVIRONMENT\" \
          -v image:\"$IMAGE\" \
          -v locations:\"$LOCATIONS\" \
          -v vm_size:\"$VM_SIZE\" \
          -v allow_ssh:\"$IP_ADDRESS\" \
          -v test_args:\"$TEST_ARGS\" \
          $TEST_SUITES"
