use test;
use http;
use core:io;

private Request parseRequest(Buffer b) {
	var r = parseRequestHeader(b);
	if (e = r.error)
		throw InternalError(e.toS);
	if (o = r.value)
		return o;
	throw InternalError("No data!");
}

/* GET */
test ParseFullGet {
	Buffer teststr = "GET /resources/hello.htm?banana=true&request=big HTTP/1.1\r\nUser-Agent: Mozilla/4.0 (compatible; MSIE5.01; Windows NT)\r\nHost: www.tutorialspoint.com\r\nAccept-Language: en-us\r\nAccept-Encoding: gzip, deflate\r\nConnection: Keep-Alive\r\n\r\n".toUtf8;
	Request req = parseRequest(teststr);
	check req.method == Method:GET;
	check req.path == httpUrl("www.tutorialspoint.com") / "resources" / "hello.htm" & QueryParam("banana", "true") & QueryParam("request", "big");
	check req.version == Version:HTTP_1_1;
	check req.path.parameters["banana"] == "true";
	check req.path.parameters["request"] == "big";
	check req.getHeader("User-Agent") == "Mozilla/4.0 (compatible; MSIE5.01; Windows NT)";
	check req.getHeader("Host") == "www.tutorialspoint.com";
	check req.getHeader("Accept-Language") == "en-us";
	check req.getHeader("Accept-Encoding") == "gzip, deflate";
	check req.getHeader("Connection") == "Keep-Alive";
	check req.body.fromUtf8 == "";
}

test ParseRootPath {
	Buffer teststr = "GET /?key1=val1&key2=val2 HTTP/1.1\r\n\r\n".toUtf8;
	Request req = parseRequest(teststr);
	Str->Str params;
	params.put("key1", "val1");
	params.put("key2", "val2");
	check req.path == QueryUrl([], params);
}

test ParseNoHeaders {
	Buffer teststr = "GET /resources/hello.htm?banana=true&feet=big HTTP/1.1\r\n\r\n".toUtf8;
	Request req = parseRequest(teststr);
	// check req.headers.count == 0
}

test ParseNoQueryParams {
	Buffer teststr = "GET /resources/hello.htm HTTP/1.1\r\nUser-Agent: Mozilla/4.0 (compatible; MSIE5.01; Windows NT)\r\nHost: www.tutorialspoint.com\r\nAccept-Language: en-us\r\nAccept-Encoding: gzip, deflate\r\nConnection: Keep-Alive\r\n\r\n".toUtf8;
	Request req = parseRequest(teststr);
	check req.path.parameters.count == 0;
}

/* Query string parsing URL-encode */
test ParseQueryParamsURLEncoded {
	Buffer teststr = "GET /?q=test%20string%20%26encoded HTTP/1.1\r\n\r\n".toUtf8;
	Request req = parseRequest(teststr);
	check req.path.parameters["q"] == "test string &encoded";
}

/* Cookie Parsing */
test ParseCookies {
	Buffer teststr = "GET /resources/hello.htm HTTP/1.1\r\nCookie: sessionId=abc123; theme=light; rememberMe=true\r\n\r\n".toUtf8;
	Request req = parseRequest(teststr);
	check req.cookies["sessionId"] == "abc123";
	check req.cookies["theme"] == "light";
	check req.cookies["rememberMe"] == "true";
}

test ParseCookiesNoSpace {
	Buffer teststr = "GET /resources/hello.htm HTTP/1.1\r\nCookie: sessionId=abc123;theme=light;rememberMe=true\r\n\r\n".toUtf8;
	Request req = parseRequest(teststr);
	check req.cookies["sessionId"] == "abc123";
	check req.cookies["theme"] == "light";
	check req.cookies["rememberMe"] == "true";
}

/*
 * POST (FIXME)
 */
test ParseFullPost {
	Buffer teststr = "POST /submit-form HTTP/1.0\r\nUser-Agent: Mozilla/5.0\r\nHost: www.example.com\r\nContent-Type: application/x-www-form-urlencoded\r\nContent-Length: 48\r\n\r\nusername=bobbytables&name=robert&city=EAS&age=39".toUtf8;
	Request req = parseRequest(teststr);
	check req.method == Method:POST;
	check req.path == httpUrl("www.example.com") / "submit-form";
	check req.version == Version:HTTP_1_0;
	check req.path.parameters.count == 0;
	check req.getHeader("User-Agent") == "Mozilla/5.0";
	check req.getHeader("Host") == "www.example.com";
	check req.getHeader("Content-Type") == "application/x-www-form-urlencoded";
	check req.getHeader("Content-Length") == "48";
	// check req.body.fromUtf8 == "username=bobbytables&name=robert&city=EAS&age=39";
}
