/* TestOfTiger.java -- 
   Copyright (C) 2006 Free Software Foundation, Inc.
This file is part of Mauve.

Mauve is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Mauve is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with Mauve; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
02110-1301 USA.

*/

// Tags: GNU-CRYPTO JDK1.4

package gnu.testlet.gnu.java.security.hash;

import gnu.java.security.hash.IMessageDigest;
import gnu.java.security.hash.Tiger;
import gnu.java.security.util.Util;
import gnu.testlet.TestHarness;
import gnu.testlet.Testlet;

/**
 * Conformance tests for the Tiger implementation.
 */
public class TestOfTiger implements Testlet
{
  private IMessageDigest algorithm, clone;

  /** Test of variable-length nil strings (0000... etc) */
  private static final String[] ZERO_BITS_ANSWERS = {
       "5D9ED00A030E638BDB753A6A24FB900E5A63B8E73E6C25B6",
       "AABBCCA084ACECD0511D1F6232A17BFAEFA441B2982E5548",
       "7F7A8CE9580077EB6FD53BE8BB3E70650E2FDD8DBB44F5C7",
       "605D1B8C132BF5D16F1A8BC2451733F7F0FF57FD5F49E298",
       "7F3A0954A5C374566C72370D1D97C2AC8FA9B1E3CB216E31",
       "B333735AFDAB30B4B597CCD137AC1AD2B30D4A047BAD0127",
       "1F27BB5926F35CF5D52A24817BF56CD17D79BB65ABE85785",
       "5229DC51A494B913F8E04C4C729C93CB8B260CA4EE8EA9D7",
       "5DFE02959A64534E0E679937B712F7DDE2C7B982597D315D",
       "654D00C774FBE74CFE4C0A6DEA5917823A6B20070CD0CCCB",
       "C82A6419844FC7028B52CAC8186D63B85121AAACBA622A87",
       "06F6D65EF081AFE892AC0946F275DBAB9D3B6D9C55E517C9",
       "42C40537A876D69DF84FF36FBA3127901B112B9E96688012",
       "6FA00EB82505EBC4707F675B0A72A9F9795F647B550F155D",
       "239549439CCE539F304CB7A95160A033529B228CA406EFF9",
       "464B87921CCDAEDBC0D6941610D1EB19E536036096403F32",
       "F29CF9B53DA4F9955E7BB766EDBAD59EA914E5354DDF9ACF",
       "E9B6BAA124BA93A4501DB9A92F212A14C538CF82C17E89B4",
       "03EB6D4703EFCF2E088515D3B902A1CF78711B7E7CE3AF74",
       "53BFCB1EB5CA9B2E8A4225E6C00584023FF8AA0017AD3A0D",
       "67F6D2B74B13251432469B9CE6A7C91EAD6E72B9CB51CEF8",
       "81A68B237DDC60C9C31B34B52A754A8A7B1F51D40781B988",
       "E99E18449AEB4421CEC31D1521C23445BC2598EA5DA14A09",
       "CDDDCACFEA7B70B485655BA3DC3F60DEE4F6B8F861069E33",
       "424CDDE2E5478A29A71352AAE538EAFE9F1B6F07AE05137B",
       "996B4DD292782A19D742B534916870023F32E72F1C197625",
       "2C4B719E5E0BFBBA616FB8AB35470D84B676C7D3A6395262",
       "F178E06FC3319BA18F2EAC3DCC5240D06BAC062BCD5CA1CE",
       "E899374C34AED37FBC9E0E4A0D37A13045639D5CAF24095A",
       "3D8E74F26682750FEEDE30A305C9CA162C6BDD9B790BCC5D",
       "3EFCD915BB46020B8999C9AD56356FB91F93C4907C4012D5",
       "739414BD4CD6AB967CD46A1D943412757D858B24D1C4ECF7",
       "3615FE2BB3B94A00245A1B3DF68B9D43BBAD82B16BFCD6A5",
       "9E9AD6783D3775885E65D05EDA4262E179836ACBB9867CE2",
       "5D91C8314556534FF3DEE0515334F09E16935538E347C029",
       "E48325D26CBEA5C14AF5D2BC043242DE2A5DC17EF5798BA0",
       "0DE25B928CF3E4FE85F46C18334D9C9DAE995E4889068C4E",
       "1BD4360038C1282E66538B156E9F604AA0262D4608184A5C",
       "FCC72686AD012FF48D1471CD0EAD606EC21A5E496DB66103",
       "F11D39950658DBDF786703E3EDAAA3A654B4428AD419DF11",
       "2AFACE18DA68984D907D1C0F647E9244D6976F4F471064B9",
       "E2056969720C08B5C9C8F903416668A7F5B2D6F95046C5FF",
       "D95BC879A68D549A5296BD34344C5ACC4D826E54ECEC346C",
       "D33A7246729743643BF4B438FBB9BC2374F9E346A0868DC6",
       "1201A8247225F31B928390B35787C046653249D0441F0AAD",
       "1E63B6124EF57A92B36C8FF837C38CC9072048ED98EE853F",
       "F2E4B6637FDFF18F785894EECEA514E7B96B2409E62DF3FB",
       "10DD94B66BA6AE0498C9C7754844662E5D8B62E27D2C4D26",
       "8BD37E3CF05E59537389672DF921392A45CB57ACFE9247A1",
       "41621DF28B038C0032508592F986C7C4832927CD1134B2B4",
       "978365EDBC762C10187F212BF9115D79FCBD489EC135AC9A",
       "8C4A9C341016B2D74E1DC75478A20A57EC934C38E6D60DF3",
       "FD5A1DDED7D38B987B582FD91E95FF8ED84DDF5D486A00EB",
       "F5845CBBA386319361D4042658606947FB96D72934378AA3",
       "A4EE394B2A208E9B0A74C6D57568E470F6E658C44689FC63",
       "19208AEF976EEA1A1296AB46BB8519E4E35CC3D26D2B574F",
       "A0D74C6090806AE1B4A9A676950B940DC37D28F762AF66DC",
       "DA1DCA0F6E414B6634ECF34DE132F3320BE35AC8D6AB0BBB",
       "49D948E0A5149A7B41F73C30B62FEEDBABA7DD9FFA9DB383",
       "95475FE7F5016A6FBB175EBF3B13EFE41FCFE9249586BB2A",
       "8AEEF6FEB5BF02F8ACC6430FEC5CC7858EEDF9C1DBBE8F09",
       "0E425380B928052574D82A3F604162F3021361517560271F",
       "A857DD168B22B65A6DD2EA8035C4EDC4B890453D14BA6052",
       "33FF9966DDD692427A9BC4D611F3C74CF629A0544A1A7ED7",
       "759D79778FA748F8E828A568C45B7E2774A6052E0A4A06A7",
       "93E4986EC1BAE0334662495E3261769214DFAB96C6567A12",
       "D50D44D48A7519B21C7ACF59DCD56066C7C328F680F2FB7C",
       "6C240CDCC41A5DB56F7A88E25A04A2F3D10D33B6908D0F4C",
       "D8581CF122C23E06A9A8781E6635113F76CBF21ECC520BE7",
       "1E763325671BEE3E17FD894D7A4E1405315004563BFAA368",
       "57B581CBF0EF91E38F4A306FFB3791A0BE6E1FA96BDE0703",
       "0870537F7FD0A5F600E2B02B31878C2244F2E76C8D0AB421",
       "6F1D28AA4EEF347CAE41FABBC528AE71345BAC83D6572BAF",
       "704CE7D9895218DAABAF9BDF944F7901571D0C24873984BF",
       "1574D6AD7CABC7DFAEE1520289E0ED8B26DC6195C5F588BD",
       "257196F102E891E04ACD046CF099A2191ACE7F878B1C5CE2",
       "359F480A3475F89093F33289C3EDD28867C0E0F11AF79939",
       "8CA6EF62DD4E1445C729E3FAFF0B57DF5ADA90D714B906F6",
       "D917B8C5CF18FBD88482F6754EFC9D308EBCAB912A609D5E",
       "B3950A9D299A5A732CC0841F1EFAE62F3DB20A707B98F3F5",
       "4112B496A4BA7C67F040A48A30C3F48496FDD3AD2A4A9E6B",
       "D663C288232D8512986F0C2F37F20A764AFABF068F44CE62",
       "4B0292082184E5727B98ABCFD57197EE14AB5893E2CDD370",
       "9EAF05D45A302F909E6850E28AF8A88E51799E20EF75D0E3",
       "AB8FBE9C09E9906200201F3AFCEF79AFF29267BA19C63A82",
       "DB76DF9A0D2E9D21B57EC44C99653F59348981467EC2476D",
       "0C2CCBAEB70C5C1F85FAFA01B1719F363D902716810F582E",
       "FCED365E31C7E4F3B06CB6720875A33E1AC45D047D071342",
       "2FEDFD926783A5C91D51BCC3228F2846243B95E5D858AF07",
       "C1789746C0E04E42F2CEAA374FBCEDCFCB17AFB816D3D9B5",
       "C055A3A9EF84BD96AEBAE31C08B74BA22BD2489C70D25672",
       "AE1AEECA1373D4348E4EDC2B7F393A8F44A351AA952AD7F7",
       "60CA04458514989969B5D3A1537F226CB54E3FCE9CC9B134",
       "20E29A4A4C36D7C2BADE368564765EE0353709CE47D0E36A",
       "8CE7840450B26DE74B798056D30584DCC436AD79B061729D",
       "05D2A28308C44536C591DF21B02AB542AF982B81A4C5E129",
       "196ACD3484E2F5FD51AC0CC667596AB0E764936C37A345B7",
       "C4F29A86A5415A0086FEC47CB3D46A71E8D224880657789A",
       "C5AD956C7D06EC8651665176733FB2EF0D01BE7E81E0F775",
       "A5A54E12A9538A158E78AE09896DCB2CE31F14150625E615",
       "239045220EF064D13637D503A6079EF21B5F2D02CF98FA5A",
       "AAEA04D2B11931D58016A9B68B3FF543C7BFD87823EA6ABA",
       "37838610BB3944FB3DFBB14B95A4007705B3773148E58AC3",
       "3DE1ECB5FEC022C7CF8B4CF4AFFEE41D85648EAEF6B0F5A7",
       "40224B9C7ECBBB69D1A5E0A3A0F249315AE1E7DE859FD763",
       "7EF8F229B824EA82F8E8834E799A67A0B946086BD9AA582B",
       "2543ECA4058C02EFE62E2131E9A7627D6DCAA6C51A13A517",
       "51E2B3A3D56D88BDB4F77CEAC0D146B08C269EB50D7914D4",
       "9A3A264F48D2BC4B08671B0D6EE58B4E0E87D19CFC561738",
       "F8F8F7095F7FF1111811FB2B63F431B529CA740EFA9A3350",
       "E0BF4F81FB633EBF0092E6AD4EC8B6834A85DF4F639F94D0",
       "114BD7F8FCA026BA9874DD79174EA1FB7275B8C7F2042871",
       "BDE5124EBA58CD1929DD63E9A4A987296D93E78378BA7E21",
       "FAA1FA47DEDC888043BC35BD97288C3E3E8702E3790A5ED4",
       "4D0016CD63CCB3C917A9CAC0A656ABF1BF1F79A42D9FB4E5",
       "1D9C9AA4BD15F3B6DDDE9771259AA550F17CE01383DFC3E7",
       "D70BCAA1810D6213A77935D751F52C8743C809C37DCB02CE",
       "9759AE86B5CC71DB9406A2ABDD0537940D717E2953E4573F",
       "A6452EF736D5CFF79BC3C3DBAA7FC7B4B992AE7DBD87650B",
       "FB7177FFE0E718C91348A8302C57129596149F042FCC99A6",
       "A6FABD413D9FAF90AC0A8D97F82B1D28761D9B3A7A3A2E2C",
       "93E56AE559EC7BF41251B6C84700465BF78E0757576679B3",
       "24F342F67741BFD5CCA5E9C5C0D902CB6F93DF12203E891A",
       "BE7803AE8A8C84B7567352C2E3A78BE424784406D33F6930",
       "43A59ECAB48D3784CD83B0887E92D208A0E7CEC6DA3A7146",
       "7D16DC5E00379CA0F13FA261AD5FDB56449ADF9DAD3810E1",
       "00A569CCADB17662483CA36230BCC6956BA5C1D5595C044A" };

  /** Tests for 512-bit strings with exactly one '1' bit. */
  private static final String[] SINGLE_BIT_ANSWERS = {
      "B99F2BEFF1E64861ECB3D91B04C9BA4BE6FC0FE7CE9653F3",
      "FBDD470EE15111676B477532D293019EB182CE9EF17C0B59",
      "D68BC840B64746A212810784CAB2BBDFCC49FDC743AFF5D4",
      "D6B280F151C286E2B8A78B3C80F181E192BEEC0A339F0453",
      "02FDBD79A641FDB807DD80699019BC97DA69D692104FBFCB",
      "E30F39FBB7BB8C191E2CD9A970EC806EA64825E4CCE782D7",
      "0521BC64D5C188320CB9AE9D2A89BE291352F77C3E44772B",
      "60224F0B05145D00FB3C09E78522FE17FB7C64AC90AF4BE0",
      "3104BE53E8A19ED90BB03DA1FC5FBD052EE7D80B4AB31AF2",
      "ECE34E9D1BBA44EBAAF18254812DA74E1B250D874049471F",
      "7FBEF692D8F3DB59912960E1C4F255D65CFA93E64CDA00B3",
      "180C2B906440036CB48ED7DBD670CDD06054734CA9E3E531",
      "EC1C85F1C4361CCC979FA94565CE47BA766B26C03E0155ED",
      "0CB103FE2F16046BCF473C0EF5F30510094B327A811303AC",
      "35E499F9FCCE1B0CFE442FE493333763C8E5F956A255DDBE",
      "E8CFB0B7651DCF73891553FE1AD3DC0F1BF5CA89E2243771",
      "9F375EFC406EDB4DC7008C6ADF57CE7BC6072BD39593651A",
      "C40960EE520A3F6E38938C4D774695FAB9AA99ED98724686",
      "C925142A228B90DC4EEE8D740480095B72EEF71717AF230D",
      "49F67D2EE9E5E07754EB3FB587A5DFC9A5A5F1AF952ABB90",
      "8EA4A53FB597B8612B751B25B4D67A3211BD9B55810830FB",
      "7F8A9779678C20D7B26126CE54CF40D586D639038E0D898A",
      "45B2D2247DF2BE8762F7C2BFB1D282DB99686B8C61F12DC2",
      "5AF3F43A65A123F25318B108322F1422FBDD4E2088B3FCC5",
      "255BA44E43BB186AC95A8BA873F27C17D4209A7CDA0743E8",
      "6CC1C5FE50A6A3E1A8C9151E529627B7825E797CA2996763",
      "4C8E2574514F1908500775D706E66CB5E098A07763CB29A8",
      "80E15B5D7DD082786B73BFA7C11ABE846CC08992D82082EF",
      "9AAEDC20A2B5C9B3AC1FEAB011E0BC63C3F14A77716745B4",
      "20FDDFC28496F4B7C474D1DC0D2556B71A0BBE0008FBCA74",
      "6C5437A6A7ACD3882300183FBE7B474CB7C0E45DF4BF126E",
      "3220E4DE2B99FD792057CA379EC8E0FB9277B2B0AFCAF7E1",
      "5605262344AED0F46F6AA94AAE288A936B3783D7B0C96EBB",
      "79FE05ED00EA9FBE5C5A411B7BE9D0AB7E95FCAC4944FB5A",
      "5632B77A86B06F60FB38F6F8D7B8ACB076610CDB17C45F3B",
      "6F1C90B7CE7D352A1EAF908EAAEB69DE70CFEA924FA1F322",
      "B0582D1882AC0B5C534293ABB751072EACFCBCCD7760B739",
      "D56CC1F5522E00EF7F25DA7C524CFBEF639E04ED232FDA74",
      "FDD3E72BD41951FF6CA729207F9EE320F277C240C5ABA6CE",
      "25C341EFBF8E0197DD7AA0D098BE9C262EA24A5ABF1956DC",
      "3D8E11219C1F4A655DE818DD5BAE94724292D1FC1F2C0451",
      "78D0C2AEAAB13E6957111303BCE092987F38B125654AEB06",
      "76CE1A38D5408FCF5C60F231F8624D22EBE4F3C2A77833CA",
      "A1478F92ACD8AFBF1383F566E8074249C576C6C01573C816",
      "EEF978C6730A5A51E0C67CA0CD591718F48E586924DACAC3",
      "D244E12581769DF1260AD3B9BADFEAACAEC10F08D0BC20B8",
      "6C88A229FF9769C257AE208427684E2BD2F2616FB11DC61E",
      "F9AA7D877F5B192A751D6956225FBB166E70BB5373EA03B5",
      "8E5FC1110E41E417B771F4860BD7180A8771EC8BE3F51371",
      "16A04075744A70693EDC2823E6D3F53F8BD7DA82F61C988A",
      "DAA5DCE2246AA0B16DFCB5A6DE1148A4667CF130CB0FE57A",
      "7935C3EAC77062032E4DA41CFE266DB7B1CEAD6A92F36531",
      "12FAFE2DD9FE0F928585DBFFDF2FC940329F4CA2C428FFA2",
      "B5AF46B22173E04EC7C73D201D230E76AFF7E25E7C64C9A1",
      "04A39346609CC870A34BFE4BC381391E5096EECC0420CBE0",
      "8E30346B3CF6A33A5DF960349E79E318A559177A10132AE5",
      "E010FE9A2A139AF552D7A22D5D04D59ABE9F26099565B139",
      "7E3A17F0489C5D1CF249BDFD6D839D17797E5F91655061DF",
      "DB0EDCB6F326FFC5EC6C23A9B23630110BCF3EF1623090C7",
      "0E878D8D2984F9A638DFF1A309EA4B436D24FA40B3BC5F4D",
      "AE09F204A5E6EEA22BADD0042D00D48E60643343C951C79D",
      "5411134E8112EBD21A89CA726D6C3293BAF1522031BFED57",
      "1CC1D5B8489F213F6A0519811867CA70AAB68B93768B2EAB",
      "A16D7299661CC6978AE949CF9F607E286913CA0615E12CEB",
      "9A2A8EE8963BE88DF83D5761D490FFB0450E9C450542CB08",
      "FAC9C5FB38BF7CE0902DC3E447D3D4D6DD369DBE94A8AB67",
      "C430CAA3FD05B674624BB549B3A565C9F1A93D8E4AE429E0",
      "D03C16013ACB204439B0B84C5EC0507C07EAD360114BB6D7",
      "44F6F5EFA89F2302B9E4B6710EFA1BD79F2589575F57B7A0",
      "C392AF0E93D619A0FAAC3FC2EE960DE9464DF3F3B87728D6",
      "BF1C5CF1B59A39C9EFD3D46F9144E7E7CC75F53C83BE87AB",
      "F4DEAFA84B6859F0B32D04DA3CF8DED931870EB8EBA086B6",
      "68B3511E9A930CC1E465DDADD3813E79E5235690E279A53A",
      "D98B1C76BF4216804783DD2A082F5722AE7049717AC6BEE4",
      "8FE146611B75E5E6527F0E2516F09090FABE1AE0F51D9965",
      "B6F4518C9B5A082BA4951EEFEE8991D2605395D04FFD9784",
      "0C4C268E0243354A31C576B053FCECCD320D864892DD4452",
      "4FCBCA86DAAD5EE31427339F45DB698C56F4EF5977C62346",
      "D63400C91F46CEB1F73427A1DD61920E5D98250CD7B21D61",
      "2002BE33AA6783F40109E747E27231C652FC90B304F91218",
      "A8FEDBD8A393C41857EA09B457FDB42078049B3F269C1431",
      "D2C06B7D56774863AB6F3A9F8FFD79FB56FF130906B16596",
      "5D8339715D342E8DDCB7CF18F4878D2221BB05FC4B2563BD",
      "06DEF00E4151BBFA011D87096B16C09A955B1059EBA03393",
      "36386BE0645E7577BFF63BAAC50532912EBB7026B120E472",
      "8E4CAC8FE8C09463A5726B0FD3335A09E8971B498CCFC14E",
      "45EA2AE4AD4B1E0C6958FABC69939E1A19300E28946638E8",
      "7C97246A7834ECF7074DCAD08A8B4BF3E8A68157889B97E4",
      "0F040C8CB0C777EE04086DDEAC53751D037FC41A62DF3D64",
      "011A9D0417A070DA0D425E835F03C68D0BD8762A7174C0A6",
      "76B90E3049F98281369B0ACD1D2193EF2F63A25F75D9FEA1",
      "733DBC18BB0821CD1B33210BD9018F2EAF405E0A281328B4",
      "4C3B42FEA3DA2B23AE3C723B22522AECC84BB1D5CD2F9E51",
      "77396616E87FDEEBDB6276B30422B04C2098F85F0085371D",
      "6BD882666B2089F5A3904F91E26B34263BEF463456486ECA",
      "1AA8BC49C40EEADA030EE802C336D83D75C3EEECDF365AB2",
      "6E9615A0291AEB33D37FFF3BDDAF628178009820457F0737",
      "F78CB6642756E3253A46FC55E5C852B5D209CB886F1953D4",
      "3D900B55C9CD9E821512415B734B06A28B8EFB0488F9EB4E",
      "1D534936BE6E2831E83661E0568B170EF52712899E2F8A8C",
      "68D08DCA280F71F6E33FE81F56BF5D98BB498B720F8294F4",
      "D4D1339C8797918AEB4000CE0B75EB741EEA852B7E5FEF35",
      "F6204099D4C1B7F82BF0A1364A6322C3578A46F354A68622",
      "FC94F4DFB37B4674843E0CD9DAFDE149D89667915DEFAEFA",
      "F64C25ED176B388E022A3AA4622975F1CEC7A9A359D1D28A",
      "CE3230D5A7B57E82F92D8A36485B0EF70B3A8430391907F8",
      "D9AC7B94226A60FC7D99CD796FD5B6EA36AE798047E1B53D",
      "8A9F12D8D9DF81E43E9DC038FBA927A3A0A0D6FA43749D2D",
      "CCAE282196A07B02E896FDDBC71616298BBBC436BED3D424",
      "09503A7A95B31206D16FC3A3A6730CB379F413C255B0916A",
      "0D799B12AF89A92E0E91237F37236BA484B5192B468F92B4",
      "04A3597758A82EE3EAA34F5B0FB1198A1177B3B3B8DD583C",
      "BAC0B8BE5A41BB8186CABAD7A32F17559DB8579D2A32ED35",
      "96B5AD3931A54DF4C0754FFD17E2ED27EEDBA480D7DEC700",
      "4A3E600FE13F444B31E03CFA0863F9E316F83F68F16D509D",
      "B5DA026AE3BFBFAC6D2AB00A16ECFBA40130E1817E26E6FD",
      "635799A28E0E31EF6DA157BE71327548E1EBBE93D39A40CE",
      "00A7C94AC7D6DA73AF3537816D7535F6501C81E9D0B70C30",
      "CEA02CEB43D209ABA6217234278931ECB509813A516C8770",
      "460F8608C1CE8A7ED701C8D46D800E5097DD421321A595BF",
      "595B7A837EA74AE2A9C2A4DD9915C684E4003A78C908EB34",
      "40A34377B74D9C7404E69898DA030F47C7B4F5125379799C",
      "011A1626ACDE5490D40D6D3B212E4296DBCFAA310C0D0A7D",
      "0C38A676E739B11BF2B5F305DFC38C1194B67D5BD9ABDD3B",
      "B2BDB49CBB6A27D90DC43CAEF626531E3794B02A4B557363",
      "925F73F9BF011425B3C855EB93FF2B3702994957C90A6E71",
      "C5438E4F6F20FA52579EA70DBE976E8ED0CD0CA79F0FF667",
      "9C19CE6E144C81EDBEFA42762CCE7687D82D90C80E6E4DD7",
      "F6F514983734179B56A2DB5810D2FEF9FBAAE792DF757EE6",
      "47EA4D78437E145E56490DD5C47788BEC899AB249A782AF1",
      "751FCB36A8B567673E9571B4F359CB7452913DE68458233B",
      "6B86FAA0426879F9C1A6F4A9F26FF19DADCD0BA2C34273F2",
      "8ABD32BED2814C6428812B67CB7228BF02AC80EC80C5771C",
      "026EE4B31888E474E20F1627B30475FFEE90BAFA2E214CE2",
      "6EA00649517124D6598C2CBB109B8F8AC445771A9550263D",
      "9029BD1306510D7BFCEF39B7E3E6F3A62993A810141919A4",
      "802BE35C1D0356AA8D7E7830C1426AA9789CF9B1B75C7145",
      "78D1FE031819E727172B1FB5A0EC87AC33C494C1E37C222E",
      "E87A76683F7B6D8CA2B47C8C41F346764053D8841FBE610E",
      "E3E18819D97CD24E6C82AF5212876375134D9401E1CFB25B",
      "528D81D4D77B3C9360D1CCB3D8D906FAD4BAEEB8C417AF8E",
      "940509DDE91065297F51C95D6D61196BCADDFFBFE43A0490",
      "1F745AD4161EE006A4260D7249E0B68B0453B01819EB2A14",
      "B1AA443FAA0B47E51A307ECAC7FAF07CA8732E715513C738",
      "CB30A43A2DE29002ECE22BD0CA7FA5C40E0F1EE3C9784249",
      "A5B6D8400303E5118489CE082241808027945C143D1200AC",
      "FFB82281B7B2CDFF2E83440CFEC036D03C26BD5133AA56C8",
      "08F21816D77A3D5F7DFFFC5477C4EB78AF660D7C32B1B6F0",
      "286FC4EA0BA3EE25CA28CD635DA1077BA00B7BBE1D319C9C",
      "ABD221DECE666C1E0ADE89ED127001B1AE8B37A3249BC237",
      "EFD037EAD7168C5B48E18DBB0BFE94E7A7337ED9F9DC0B60",
      "17A915BB90D878000A07BBF2F7C29605541CCBB2BA33BA9B",
      "EDBA011909174DACC7276FB748DA3D72037688E09B471CDB",
      "18646AB2DFAC64E3D1B8298F9DA48F8ACD3EC8E9B6C091D2",
      "5BB55C940C48C5F173356ED2F76F135BB33C49C6FF401CE8",
      "444CCFDECBBCAAC55FD538C84FECE4AD05E1F86CA2183046",
      "0DD2986EBE58B62051C5F8558B33B2150E2162273A21D2D6",
      "BBD0BE785D950FEE7272A6A42E20EEF9FCC96DA0862A9D4F",
      "C6EE3736BED22F09A3082957D5D2D9939E926DE56B23E540",
      "F272AA5CC30126CA15ACF94904EA44A28068BD3B3ED74BDB",
      "24713F450026D6D9717475558F33A84CEC5B8EC74367A1F4",
      "021A9ECFC35A18BC91568F76A06EA7B6F068B1B45E725F2C",
      "54027E9B61191C3ADE350615D809EC6A4BF7FE926946F93B",
      "27C362B26734609E760ED30E492273A2EC0786CECE1B62F7",
      "C56A7C83A9F780DBFC564564D60B110894B3462BF202B31B",
      "E46DAE97BF1D1433B91E7C4B30A3D11189BD1959F363C8AA",
      "DFFFD3EE84CE33EDBD6CCBDF9FDBC4C6CD9F6C3663AB3124",
      "CA9D313C0A9D62198C2528B856627BC9C70DD1C6523AA09D",
      "DAB676CBA9ABC9799E4416E703CD4C89348F3513D93BCE58",
      "383C379EE484CB248AF0008F339F2B76C25DB122ABE46C2B",
      "466C5466009BFF5B4D9639CC2DDFFBEC31663189A11A3F06",
      "5C2457FB54624D291D0223FDCA7CE3EAFB85B6D492408A9D",
      "FE593DC31045C782EC6B52216FC64B35FD33EE3904BA9966",
      "D7174534437E6168C2EF60B0EA9DC61ADD8098C0694539EF",
      "75FDFFFAF4F1D44DD610CEDDB215E16593340EDE583C875A",
      "36EAA91B577FB20178B0F06FBF369D07FBCE8445BB36774D",
      "9068E188528160BC3B2C2036B4B6140BFCBC98A031847132",
      "42E86596DB6EA0724CE131CC379F5A3F37646C87026DFB87",
      "733642DD7D4F08A2303B7517974AC6D9E6EC65FBED56CB05",
      "92ABCE48F750C364C5FC649E3E2168E5BE83F6176A2DC552",
      "BB841126EBD38F36A63FDA47FD5971DFF6D9CF0432811535",
      "1A5F8B7F34376121B1BA35FB19DD2DE6A1FFE55D22703940",
      "A4705AA774708F6D6AEABD6B527C973DD3B955E144812BC5",
      "ADA04C7993477041403A144C15BBFEE889A6F835B9F968DF",
      "2F4179071804EC9280CD2EA6E265AD77633F5196CF8EDB32",
      "4378BD6082B465ADB4007CBDC5F764C402ADCBEE4347C08F",
      "F75B2A3267B96CA1EE4191B99CE78B079333F10EF184D91A",
      "C3891913D02202A3DFD2BFE70EDB992275744136452C566F",
      "07B465A5110B6F1D8CC50B4716A5B5CE9985CD0A7A857EAE",
      "AA3E818EDB7190F12AD58D9A4A9DAB53209AEE83F5B3C554",
      "C570CAD73E05B2F8283A26726F07BAA339EE78ED7CA42F62",
      "B48EC775851C9AE303F05AC8E51E62FF169C175792C2627A",
      "1E3047A18CCBA9F21D1E58F90FC2F8100D94227E29DF84A0",
      "5CA88D9C7D388BCF7877FE0F8CDD5EC767DEC73783B60195",
      "F0AFF7784660E13949EEBC8A817DB227D7DC2248EB0F659B",
      "9B5D9AB548CE2CB1B4F121B5BCACB724A02A655E56650014",
      "DA5D70AC87DB9AB9A19D7E79FA3820B9A6855F148A662E9E",
      "0DA0AA0A4187C2AC05B220AB88498FC244FCEF7FCC59692A",
      "DEFC6E4E0FF252A652CAD76C06429D7D902F80D0C132EC2C",
      "FB8EA1191F9272038017D3D10B0F901514490B119C3C086F",
      "A0F433C8EF9F86EC52989CC77B999EA02827EA0D57F00766",
      "73C51E27F033BDCF09E29C19740AC05676C7DA219C83F019",
      "C1A41E1999A6E7ED8D2407005608A44999C0918F009F12BB",
      "C1DE2FB59A6A6CC45D3E1A9A502F0327E4E0582692882E01",
      "FA371922F657875B03268090C91EF8A7D968486EFCBE853D",
      "D7786FFE8F9411E982D591C63A364483A733A005D8DD4F52",
      "EA71A84A2AE43700E2D6CBF9705809F38EB01E3AA3500981",
      "AEE84693BEA5105467DFFF30E694772F31148CCECA362D0B",
      "02C9565A480DFE315479D62BFFFF915768411199596D0067",
      "0EA5DA49521F8C298EB3A65B157D3B0E42E0BDA0EED10EB3",
      "A8F8E5533E0025DDF4421EA099D9C553736B86626649BFAC",
      "76D357BCB224F3B59DD6286A3EA0D4C2D09699E68ED7D4E3",
      "202794850A8B2A8D611473C84F18835F322D32E04214502F",
      "1BED2B5DAA818E27EACA9EA5BC6B5D88FF7605CBC529C423",
      "6FCEF79BBB8B556D597071E6E86815575C36955DB95FF0C9",
      "5C76AF9054B56A78EE4BAE2F69E54EB4A4DCCA9251EC323F",
      "8645904A860B9227CC2EBE31269BCB033D12626D4E7B1E07",
      "5CCAA2D96A4B90DD33CA1A8F63D9A32785FCE91FE9D261FE",
      "644AEF13CA34442715AF3B71F6F40863B9FE5B68D9AEA69D",
      "BB8F2FCECA6DBE2233D94CC9617C0C924251503E15ECBEFD",
      "C0CE5B9858DCAC7C2A7FD41EAD46C1D1F4A55C612EB87095",
      "C3209B535B1893D292F24AA705A54AC6893F5C34DFC0B89B",
      "4A9362D913330BE01F8AA9A40621CF2BAE0AEE998182BFD8",
      "6757B6725238FDF3AA6E92CE05242DC7A138F55A8A2B8CDF",
      "9C1EFCC39E943DB17478F55C2E8F5AABE665A257A74E7CC7",
      "9E48EA08AF2BC00367D950D2A0BE14D0F26B71FBF3997B10",
      "A6BBD6177AF40DDDCCF8CC8B8218833B8A89D6BE122E12AA",
      "F7BAE51827F6C16F31508A28D362DE9BA4984E47C88DC037",
      "65DD7FF5BB8CE8D5F84480BF5A0E35127CD736A1CDE7BC12",
      "E8531E2C7329ED60B7246E6246B1FEE13B1B70F1A119CE54",
      "7B26F57EF6D56F8966AFB3AF6C9BCE1C1948396CE4FFB1F2",
      "D6EFC3AD5E62E9055C414D01E11E0EC9374B5F4486AF92D9",
      "B4000A22B7ECE2B3BC2695A2E98A9F4FC981DE2F44A80CA3",
      "30F506642345E45D1116318D8F9849A640B13C27291F5B83",
      "E02C8DBAAA96F301081E0F6F6FF3163D11B0340C5154E06D",
      "4D176A285408CA3E69FAA2F90B980CD10BED31F3B7C31F15",
      "8BBECF1AAD25FDC805039E0B0AE53C72B2D598400FAD8A0F",
      "68213C2CB2C200AE8E130125F0DA1A81AF46DEA7A32564B0",
      "B65CAE529F4F5640EC8E61D7B18DCABF5C42D09D16638E02",
      "6D9BDF35483F61B8B41DF2BB3DAB479CE960916A91768BA9",
      "C7065FCFBD627402B557D08EE245E4B792079CBA32E7E956",
      "FA7CAC5FC22F2AF0FF1E052D28817A65966B2B0C1C2F8792",
      "204A233E044B2C07EBBD2C529A8D578A07B21BB674EE4D64",
      "5D64C8D0554606505D54BDF395D6608AE15D23144284C05F",
      "C90A9B1D92220CA0F3F4A8E65155FE0E1D152C89213E96D9",
      "3C8673A6ED62C5FFC880D24A000053ED7042329424181DFF",
      "732D68C9AAEF89D27676A87AEA0F975BC31E9B7184AABE49",
      "61D3540EF01107CBEDCA147F615B0DA929474344FA8DAB0D",
      "FBF56E336FF510ADA9A801F84E4792FD8E9B1C3B44B4D666",
      "FD5EB3F4BF2E475CA451C856B2F576FCE1CAC3B700348176",
      "E5D6F6AF3BB3D2D22975AB500DA24B5783E651C1AFADBCC1",
      "22C7EFD25977630807C1D0FD7DC220FF426D589BEBF3F818",
      "38C0B8ED87706F4A7C2DC5EADCD01C72193261CF37C196EF",
      "CAA17EB9893A85EBE650736A0B56FDCC9DDF6916707D1B99",
      "1FC43409C387A59F2702D1120CA69AE1EB8BCC072AEBCC89",
      "9D5CAD387705F97F784279C85B5C09D3A8216C3151BB8BB5",
      "DC1EBB94F3E0E6703DAA5715A7E826554DB9EDE756B2EED2",
      "5E626E0FBB4CAE926AC717EC009CAEE23B5CD3062359E20A",
      "F348BF048DB8D98D0BD0666B520AB255B12CA6047B3166AF",
      "8F929C79DDE5DE2794998170EFCABCE511637A29810ABF47",
      "FE4661EE9DD9FD928D2AF9B5898EAEAB617E88369457DD67",
      "6E0C9BCE677EA50439E02731B301300511643AC1FE00125D",
      "4589434D3BE9D44EF45DF1ED91AA97D812ED55DD869B9067",
      "0C0DC9DAAD3351B7CE17F8D1F6DBDA62F91FCF6C93019C45",
      "07FD597941C0A30A137256D65E5D2430EBA2C6EF24979E00",
      "EEFC74ECC53C334768D59EEB02B49A462A976CAD32FA9F4C",
      "292EE351E25622EF85A07C0C161FC0FBC09229C176998D35",
      "7ED81D7B5AD71E60F04E388D2E29E044EE253D701D213ED1",
      "56A2F7E6F85975BFCE0705A124AE4CBF7C3015690304521D",
      "B07EA50183596A2430F3DEC13F96845D1070A3D32F5EC066",
      "526F6BBF1EBE525ECF7EFBE693A06B746307726F2011529C",
      "89EBE9219A9AC412BEC88BA321030B4AAA257B6410F7719B",
      "ABD5FE2E6DA870D5B894B35E72E41D590A01E40303524335",
      "1E3D537D9580A74F2D7FC71EE241923AB3848D11A1A5644C",
      "F580664AB99A6B50BFA21019261D91B3A1238FB129B0797A",
      "2DB79D2513802F939D4608E4FFE38D1CD5FFE7B667CE3EEC",
      "13B1F534DACAC2836C67B4B410151AE550AFF9C977B6A3D5",
      "86ABFE6988DFE9D6C9A884942946BE74115987EA3C6AF477",
      "9FB58A6C6822D1F7BF3AC52BD210132FD6DD1E823D3CE95C",
      "1F5E0C29EBE8A28B934E7932C0D120F2F01CFA87DC089E0C",
      "BEA84869592DF1FD12127F891CA9DAFEE767226868F878DB",
      "6FC0DF95D191F492399AFC32C4F536F43C14E189DA93396C",
      "33E9415EA0EDB2BAC1DD210B9CC2F5D7F46B5B07C5EFCC3F",
      "CD933BE80058AFBBC9FFD6590CA2F53B3696E7F86CBDE8CD",
      "15F89B44FF00A4CB930233FADB49A824F0AB540636A0E8A5",
      "33C345BFFA20C8F0B92AC6F3F063E477F1CDA525C35E68AB",
      "617EE74C600A5B9D5C591984222DC000642FA97ECAF6948F",
      "E50FF576A78A5FF73038F307740FFB190F405AAF65909C7B",
      "10F16B3B9BE9762F603DC156A2626D0A92C49D6788433292",
      "65C75607C3085EE0DF99AC88FCB71FEEA3E95D1995027B90",
      "A95A236CEAA6FE6A33DBD6B2FF2CC61DCD06B576D34D8BF9",
      "94701D90637F543B83AC736D448D915AB56ACF688AA2A9BF",
      "C005D8C2069CB7F291C7214165383ECC418999FADF197ED3",
      "F02715BA20879FE16CA382F3F33B35DA09273F524A832CE0",
      "925F4C6B45F07D01F1033AE426E554FE30AD5664F89EB521",
      "DA89E876647B2B17B726DABF0620B5791B8D910CB9E7323D",
      "A3817AEC34558CE5C4554B4BB652D52E0FB834D66173B40A",
      "EC4599DB388F66C10AF7C24987AAAB6D02CF431602C321D3",
      "D7BBCCC448440CD6BFF93D8CA1DC309BE47E51C823B85EBB",
      "2CAB4F11BE0BB64B4F16F707DF30F465A279C6972FBCD7D9",
      "12E4D1EF75CBCA5369EB440776F117EDA8B536A35C26F139",
      "17444B27596CBC519FBC8F2B90D68D05BCF51608ACC3D304",
      "D76A8B2E986EB16CD6121CED58295F0CF5EB02DEB60F98CC",
      "164E3829B70E59570D2A5B24A0F1C67A974DFFF1EF5BF2EA",
      "0EE8B5DD7E5A285EBE11965B574517B949C3487937453B2B",
      "E3639D864D20ECB48D8C2B8F892D97A7FECF83937C48C2DC",
      "4434C996B25FE43CA24746396E50C157139C57787675CE17",
      "6D1693269BBA7630B6C083EA6D8AA16F0512F07C214251A9",
      "A1D5C7ABDDF9FA51FB5C9AE0008FA953EA83063C93E49319",
      "9BAC4A7FFF14B34F32F10EB2B6A3A61A13C255DB176D6FCB",
      "C6BD779EE2F9FF6A4ACCDA2E33FD0C62D641733E414D5B36",
      "F0D656506DE265EC28F1781DACB76D1C7575DC55C15A40DC",
      "0E95FD14633037441CD9AFE78391FFE73AAFEFE532B8F7D6",
      "AFA01B52D549FF4C8B6DC71BBA6CFA27E8EA03046B95F6EF",
      "9E63417D3DF651A3353FB6573FFAED9DF4ABD7F68AAAA051",
      "CE51254F812107B91727838B619D5C4F17FC2C3C9A95233A",
      "89F63C422BB85635F432D4A6DC9F940D79E0B5E111A24F4E",
      "999534C4B3B25F32FFF05B893F4E26CF8F1819FCB5B908C1",
      "7FAACB92FEFD12045C77B4180EA95EE7E1775703E7BB4C3E",
      "2BFEAD2B7E1C6BB0F47AEFB9909E3ED961EFB39A29794515",
      "A52AE6D0221773E42FB042B7830D28A73A590562C38C802B",
      "5B3007DFB62B78955620A0F17F58B6FE55B360C0F7A9F4C3",
      "77883951F130BA75FED13621DC61BB766E475DB7751B0D3B",
      "297DEB90367720058318C994C0382B83D182110797A29A53",
      "0BA968080D4608C3BFBE43682CBBD63F246F866AB0A0070C",
      "328F20CDFFD905862A11F70B6FDB4803133CCE8ABB8D2B73",
      "74AC023139CF3557F49E2F8A674AD0857AEEB9EBA3C1985C",
      "BD588F19DC358DC37450A6689E4D7DA413206159F3DD2A4D",
      "DB626F5040AF44803156FF86860A1367B1B644B021C15460",
      "ADE8785F2F65B67ED1740F8693C5A1CB3BFEE9E3B211B83D",
      "D08599E7444393A3C5FF2493E1B6647B297708919163E05B",
      "8AA57D3CCCB899CC92A44ACE801F54A13E60A45BFDFE7D30",
      "3415A0699A0D14DBBBB2280A9C8E3C91B954B769CE606362",
      "EAD03B5ACC5C86FC8672AE14330BF93A9489A5017D3B9354",
      "7ABE3B7FD5804A23B937C0EF5ADD8869188BF842F68D21F5",
      "739004F8FC7577D0BA954187F300E441F4A99F62E0F3C367",
      "30D99CB386570F45B8F0EB75868E8EA9EC7DBB04C26D8FEA",
      "7278F6E780965FB5C02AC4AC24BD2020D2439623EEB901C8",
      "93E7E84B803152CE17491B9B9DC6502041DA4D56C636B4A7",
      "32D63F3C642173BDE55C2A7E8BB6C942420C309DB2E7210A",
      "CECC92727AAEE2FA74B90AA728F0D3F0FC5E8A42DF283713",
      "51437CED5761BC3FCF29FDE5877675B5DCCABC85BC2279B8",
      "F6A494D7241543CBFE2BB3C68F2DA9FD9915BF0A14BF82F4",
      "DD8BE4ED5FFCF836C977352ECD2CF689BD94F9CF0EB66EB4",
      "1EF1B22652B07E0039DC257B30E88A2FA5D3FC64C79C741C",
      "775E4B0447AFD1C960320C042EF2AB6C879088956435872E",
      "A40FB52C188AAC639CF9ECE72B978C812CA7991997CCC3D0",
      "F6F305D3ACA46CEFDA5FCF420B49C9C13F14C3247E4214AB",
      "2A5611447591A609BEDB9B99AF030D1548C25C1090FB7252",
      "6BB8BF7CF510DF644EA1771146EC6E943E26598734BB2022",
      "8508F80824415D97E468B19EA305B0473A915B996DC83A31",
      "8814EE7A02164290EB431E70D50490DC2313BD30EB479857",
      "EFD4A311C5F5093AB23F2C775577EEEE230643A75F7938C6",
      "6FA980DE170E4708F4F59BA46DE0EA340BC61C540B0D12BD",
      "C699821C7E342804278CA77879FFB9332F274530C673385F",
      "FCFA7BA4FE1736CA672914AC98905FCF235CB82F10BE9981",
      "BB1DC4CAA892B46D15C44125DEDF64C63EC4DE5B807F93DF",
      "520F25FEEC8D698D31C164D291A54B906564A75B7B693239",
      "772CB58B4086454862B813AB25EA5E33BE530D3DF48F8BD5",
      "CBDBBF709ADE1701ABDA7A258528D675B505678E6B643FDD",
      "A16CEC411677F88C1F751442BCDBC9F013AA1F4E829CFCA8",
      "9DA04A1EF52DDF86660B9EA7D8AF385BBE6369B10AF2F3B3",
      "049F7036F40FB51260E4B8CB0696EA03D375B401CED3E2C7",
      "EEB8208784C4C65DA497729C948063BD9F51FE5C026CBC8B",
      "F39BA71DFAD235C2E0CC8BBAE4CD8D5D1EEE9247086ECAEE",
      "8372D7953F7697BE48661E3718196082B06EB448EDF3D839",
      "D41785ADE9E2FFCAA5B9D44613CD23CD9C3A519CD9C89BED",
      "620D6396A467CB957740B85D5D51E53279DF78D56D6CE8B6",
      "89905029EF0112631ED9DFDEB63BA26235B822022B686D59",
      "51AD7F7D9611ED7C24A84525832A2F8CFAB80F907C02533E",
      "C645E4CF66DA4B343FC0E9221B4616E8CAE1E65E50195E64",
      "38EB1AAA6CEBBE8A21EBBB9C0976D29F4089EE47ED1D1F4E",
      "EDF66F50CD0F54FB3094E3412E28177A53B1CE56D0FDE4E8",
      "11CD219A1E03532D2D4D61F1563C2AAA8A1586736FEB5CA0",
      "01D50B226A3517A3DCE79495FE1410E7AB6FEEAF8FA24BB7",
      "F981D517F259B363A945D2D19447378649E597A37641B911",
      "B36A8BB7E4018ADB5DC72B4961501F9A5AD5931F5E97A6D6",
      "D78D1E9D4682A0777A6EC0D8C2F165CAB925966650937F89",
      "2F6D9783D5BF5976DCC31C7F00B9FCD04B6C16BE92417C9B",
      "22B9F7C41FCE5A4638D6E6C61265B58D18C2D5FBBE33E9F4",
      "37B7BA6658F835D817CB114D06C197D23C47708352EAC551",
      "D8E97C4A5F14A1F595C041D0ED59C4D4631E7AB673484824",
      "AC782146178A613B3FEA97DBF2018A6C7E1EE3789BBE2A51",
      "11FD098F69CA3BE809474769C7DC6664A37AA26358A1EF08",
      "93856CBBE6404B2EF9D4240556810833C73D6D049FCA47BB",
      "C8466665EA45CE4CFD88F819D5C639777640B84D3D60C4BF",
      "9B4834F18ED69D519D497AD84204BCB9D749F122A86206F1",
      "7571A4B267EB04E445803700DB6A72242E1DE16B1677C410",
      "2A6518AF7D1C26F81852CABD90366371783BE49E94BEA42F",
      "AEBDEE389CEE275743FD1DFE10716DA734049874700D55C1",
      "8EF1808D19882CCB413F4B432918984BFE61D3419999642E",
      "26B8CC222AA5C53969D3DB6DA64743B71063AE978E1F5899",
      "2B8FA32B543ED0E1A8EED1165806602675D21D4A330AACD5",
      "7BBD8EB068F2EFB116AC6EF6FE80BFB0E8023702D17EFA6E",
      "344265DD1FEEE145A37F268EF0877581998CF323CFECECCA",
      "9B606437205BD1DB249D3566D7AEE9D289C9CFD0D57FD058",
      "C28D239C59BD76719345840E7C491748FAE7FCBC5EEA6C51",
      "BD1EFD0CFD6283ACD2A95B1DAE33ACC47A14D21AE3A74DF1",
      "B0EEE8E5A9256A73A98B641F3D29092D2F1F1B75509F5455",
      "C46B3EB0F3B3C6D61CF0E5A263D4499A91D68055B1EB4996",
      "C14215E03D57E2D109D88F48500F0D2332A819F1716D8885",
      "DD38ABA719EEB2B7B78DD4455AB2DF07A5DA8608BC8250EC",
      "1175351A36F8F2769BA8A206185C10B241189FA4D6220C83",
      "208DC327BC1BE3B55EFDEFEF89776DA5EA979CA479CC970C",
      "AD80043313C929A421495430561C49F0613C195D3578B518",
      "8384691A0DCADD919AB610D69A4C478A582525D5C48B7F17",
      "2FF93D872E9E6AE1783C0EF5B1AB9510050F0276563BA02F",
      "D3C7B2D23986E191D95B19810843B63360578024F3272DA9",
      "29F56FA57392FF312289DF4A88C718FA56D2C3357FE6D0D8",
      "175884C528D5584EF0E0AAB61660EB9684242F00F6911A24",
      "F5D5EE0BFBF2186BD6CE3C63A9E19852EFF62D39E8215068",
      "6712197F2D9E424139C93292E9B684A11579BAD5546A9F99",
      "EDAFE550B5634A040E23A008DD3B15C476907B3CD596725B",
      "230EC8B2DDCE48BB1ED8928509CE6659981DB303813C7031",
      "02371A2CA23F8781DCCBA0672C979EBF0EA8F17F4CAAB443",
      "E0EFB4CE4025CD25C760F9E822F954D6B38319E71ABED25E",
      "2030785CC6D18999182F7B6A9BCA9F72F86829F89B306C37",
      "3490EBCD433D9FDA74EAAD97469345846E190540C6D82642",
      "A9C3E5AB60F1666619A899B8F3AC2370A5BF2298B99BADBB",
      "804320365682196340806665DC55299041D3BD85728BA0F2",
      "53A2DF8B0C83C7120BE2A9608F1B5A4528BEFEC9D7970487",
      "2BCD607D529B39E1429F0C54B94ED78F826389086C7739C8",
      "0C466A6ACFA584D239367547536323A32715D0DBB5867B0E",
      "2FC3EA605A33AC26E78D8CA19F98CE7B5827E62961D89212",
      "8761107DC3785DBF710BBBEDE5FEABAB6F175A5ECFA961B3",
      "F123F30881EC794674403328E3CE7C80BC95DAC12D7B360B",
      "341C9DADB601F2781F6ABEB933B73AAD103A81E8D96F6730",
      "38E53597534CCF0E60DA817C62924453E193D61861043244",
      "3DF84CC5EDC3D732948BE841302B9AB708F5AF33DC537A31",
      "B546891818224B14E90827757DE18BABD71449E57F8FCFF1",
      "A7D6E7A48C1C14CD8DC58D7EFCD4FC8D407F15A1D4AA41D5",
      "0B178B35A97C06D9D4D2A5838AA867BFDBDB36D674AF33D0",
      "8C22DFD72E5ECBA3D0E0B54985D01223032939E7B7B33697",
      "A52E67BE70E116BDF6EB61EF225DB94DF06966203DFA2E31",
      "4AB1B7BAD46AF64C8B6203A8889CF56A079F18F688790896",
      "65D1A4BF8DBA94C9752913A4C15FCFB7F18CCF257D48DED6",
      "090CFB92322D482D9D8AAFF716F10FDB491BE0FAF50C784F",
      "94F1F66FB01D719F04D36CA30287F7E534DC31B0D91CE1FE",
      "D8689D811424A59B870F84D11BFA1D202BF57896154405B6",
      "D7F2171F5B7523C54B4D1A81BB6048C99CED1E15087A9C49",
      "94DC8F6AAAD2A07CA81191B613291EB06EEC66A1B628F9CC",
      "534D75114EB28DFE082BBA4285ED30E55CBE4864278FF182",
      "F5971C8523421B68BC3D9FCA5ADC5D2C76CD0C3797232950",
      "D274B834360952A27265C12C2B70623CDCB69C4815B8AD9E",
      "D25074898E465F024B7D5AA542575176B885998694BFF162",
      "100B09B1A2778E07F0DEE2E6DFB28D032FA35D7EBAB2E87A",
      "96FE586BD7D472FFA884C5724561D517C1126957FB190096",
      "45EE36AF320341688173D7C41A7615BCCD0A431F834EFD18",
      "F73847AEB45D714CC5756DBDEE00A9E580EF91101AC8FE64",
      "30A56E8B8CDAD753A854A1BE4B61DB04F303B6CFD0D76522",
      "1BADACF65A1F790B448DF08A100E66CDB8B404DDC7865016",
      "D0DA13AA2AFC006A97C537541B505B96CE798BDDC07B9D6D",
      "FB4170AA2DBAA3844FD8485D4325BE987E2E589B58AB3B80",
      "50A0FE8090A0AA2EA503CD40DBE894004852584FDE11E3E2",
      "F6311445087C56C3CA179EF211FD32762134EDC910D9270B",
      "A292F008A7BDF88E61FF866269A6D6F6AC9A81F706D57C27",
      "700C29CF917F3634215C58B9B3B9C147FC0AEA52C05B2841",
      "2071C5313B0EA39ECC258AC29ED6E6CA8E0511C726A528A9",
      "AC6F08712123FF72E9C04D1C5C343BC2A356DBEE9223B8E8",
      "01F46A7935579B7A7F5CDD00B5E7F1D62C37726F4216DC1D",
      "820320BBB243E30FE35395FBFAF46A5C8C0CA41E4B48A80B",
      "4B3365B4AF9E0C1AB703F297ECB82F9DB6ED44819DBC72F8",
      "DF69D9A819C4E27ED1B7D66678FDEBD8BBD1747F892D2B55",
      "C0F17FAF958767C65EBA2F68EE245205CB1BB73290AACAB6",
      "D62C3EBC661115CDE4F4BD646D745ADB7225304D2712837D",
      "37E9A837E07B12281F4740D8353DD6008A077F6EF51978B9",
      "9826470A5ABEE6EE66BB799C9A9206703174AB77AA6223A4",
      "506C60EAC24C18D955364759BD8CD522157AD658C7B1652E",
      "110923F8A4E154057211F273FDAF0F565AAD0A0962C4E08D",
      "30CE0B97A6578CE2398BF04E63F78B2CAB5DC97A140426CE",
      "3AE75A6039177B44B18B2C02EE148388F93E6EB96A989D3D",
      "0DE8D7C180C5FB4E90ECF4165EECEDE97316170B7B00D681",
      "A1BA31195FF34A411C5D9A72A84C83C7BD7C6C777FC7DA4A",
      "1D7295FE7B297D4B752159B7DD71483D53C7799FF82054E4",
      "F0210F4416BE56B157EDC9A76684D1F1653EFA6B7D06079B",
      "311229D7C5AADF9CAA0DF39E441D4E119D0B3887194B9751",
      "7E5E875547284EE4952C250DA3BB4E3BBE59086FA638CAAE",
      "890371D12AF03062698C2F4F5F2D7272DF652EE99C4233B3",
      "BD3F48F09AB50132242F762A61904E945CD3628A925A8B82",
      "6401C4DDCDD7441581AFF57D13C1198FC2347D886D319FDA",
      "2446A367AABBDEBA9EFD1F8CAA9805EC1A9B45200A90E2C1",
      "37C1C64324D78E1F7549E6169FA66D492184F3A5A524810F",
      "91A620E120AB9C4949C6D3E27A856F079F76DA00D56C3A22",
      "575C19C550408D8454650AF1739812870E371BA8F34B6861",
      "5AB6FF5B263ACFAB2013C3068C03A82979EA6DB287A3ECDD",
      "BB40E88E5DA7D4B12356124D9ED74BA5283C502FC68B5A1B",
      "CFEC678B1CD75F4249D536C9399F11C12CE9938EE8CADCFF",
      "8A26E3715E0471DD3616A9E505F4C73722BB42FB233BF641",
      "ED38E1AA4B73AF6024471252F8102E77ECBF4BC315DB1C31",
      "4D5E37A37AFC8C7DBBC25AF2C6670ED2DAFB56B5376E2550",
      "958603BDA98C29A87AEDB0C1C8F559D161EA4FCBE9EED392",
      "A7F58251627E2C4C82ED1AFADB2FFF3B3483CDC279117E1E",
      "4B003F86DA13C24E10041F934DC02EAA9E86C4200550DE8F",
      "6139AB107BCD192E8419E869F77738D2624BA81FD774828F",
      "15E286064681AB50343CDFD1E178F6DAD9071E3727997DA1",
      "95A26E99BFFBB5D0027B380688D3E5C567454F4F2F283C41",
      "45326511C3070AD878172A7DA38D17ED65D17BFFB0298058",
      "C127E412181EF4734360466A38245409E44B2B208FFF4D67",
      "4C9A55B7BC50B8D7343F1FF351F13A89CA53E8AD2F056559",
      "9B93DF18074112C26FBEB1EA277070EF4D5A1778376DC12C",
      "79CE5EFD59A9A0ADBC3661F278B3C540A0332A9E8FA30E07",
      "886F0AD49176EAF30EA422CDA7D1061AF2D62DA67C672407",
      "54C7674DC49D49E63DD7296241C3F2F0B022BCC0451400F0",
      "9DFC07C61179B9DEEA1D452D3073C2A12CB0342EC8FA25C1",
      "E9C6E911956A0D7C3958F4E0252A4DF98FAE33AFC834EA10",
      "9DB1556485B4C5C67FC532C4C6937AFA3A889FDB09B055A7",
      "5875A1A391DEDD84155F3805A81EF68AD288A9E77DEB4324",
      "5DA3FCD577895036D0AB8D6BC654C59E76C0312C81F3A374",
      "3DB3B9731E66FEF6039A2723C7CD7814F2A5F8ED20807361",
      "C784C70B8F431D8AEA3F5ADF2BA55AB4BEC9C580558CD0BD",
      "B256AA37A7DA84F3599B338E2B853B92ECBA5FDE6901B620",
      "1CDEC1D10BD749090B1F491A4FC8E2F8B150D64F3215CE8F" };

  public void test(TestHarness harness)
  {
    harness.checkPoint("TestOfTiger");
    algorithm = new Tiger();
    try
      {
        harness.check(algorithm.selfTest(), "selfTest");
      }
    catch (Exception x)
      {
        harness.debug(x);
        harness.fail("TestOfTiger.selfTest");
      }

    try
      {
        algorithm.update("a".getBytes(), 0, 1);
        byte[] md = algorithm.digest();
        String exp = "77BEFBEF2E7EF8AB2EC8F93BF587A7FC613E247F5F247809";
        harness.check(exp.equals(Util.toString(md)), "testA");
      }
    catch (Exception x)
      {
        harness.debug(x);
        harness.fail("TestOfTiger.testA");
      }

    try
      {
        algorithm.update("abc".getBytes(), 0, 3);
        byte[] md = algorithm.digest();
        String exp = "2AAB1484E8C158F2BFB8C5FF41B57A525129131C957B5F93";
        harness.check(exp.equals(Util.toString(md)), "testABC");
      }
    catch (Exception x)
      {
        harness.debug(x);
        harness.fail("TestOfTiger.testABC");
      }

    try
      {
        algorithm.update("message digest".getBytes(), 0, 14);
        byte[] md = algorithm.digest();
        String exp = "D981F8CB78201A950DCF3048751E441C517FCA1AA55A29F6";
        harness.check(exp.equals(Util.toString(md)), "testMessageDigest");
      }
    catch (Exception x)
      {
        harness.debug(x);
        harness.fail("TestOfTiger.testMessageDigest");
      }

    try
      {
        algorithm.update("abcdefghijklmnopqrstuvwxyz".getBytes(), 0, 26);
        byte[] md = algorithm.digest();
        String exp = "1714A472EEE57D30040412BFCC55032A0B11602FF37BEEE9";
        harness.check(exp.equals(Util.toString(md)), "testAlphabet");
      }
    catch (Exception x)
      {
        harness.debug(x);
        harness.fail("TestOfTiger.testAlphabet");
      }

    try
      {
        algorithm.update(
            "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq".getBytes(),
            0, 56);
        byte[] md = algorithm.digest();
        String exp = "0F7BF9A19B9C58F2B7610DF7E84F0AC3A71C631E7B53F78E";
        harness.check(exp.equals(Util.toString(md)), "test56Chars");
      }
    catch (Exception x)
      {
        harness.debug(x);
        harness.fail("TestOfTiger.test56Chars");
      }

    try
      {
        algorithm.update(
            "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789"
                .getBytes(),
            0, 62);
        byte[] md = algorithm.digest();
        String exp = "8DCEA680A17583EE502BA38A3C368651890FFBCCDC49A8CC";
        harness.check(exp.equals(Util.toString(md)), "testAsciiSubset");
      }
    catch (Exception x)
      {
        harness.debug(x);
        harness.fail("TestOfTiger.testAsciiSubset");
      }

    try
      {
        algorithm.update(
            "12345678901234567890123456789012345678901234567890123456789012345678901234567890"
                .getBytes(),
            0, 80);
        byte[] md = algorithm.digest();
        String exp = "1C14795529FD9F207A958F84C52F11E887FA0CABDFD91BFD";
        harness.check(exp.equals(Util.toString(md)), "testEightyNumerics");
      }
    catch (Exception x)
      {
        harness.debug(x);
        harness.fail("TestOfTiger.testEightyNumerics");
      }

    try
      {
        for (int i = 0; i < 1000000; i++)
          {
            algorithm.update((byte) 'a');
          }
        byte[] md = algorithm.digest();
        String exp = "6DB0E2729CBEAD93D715C6A7D36302E9B3CEE0D2BC314B41";
        harness.check(exp.equals(Util.toString(md)), "testOneMillionA");
      }
    catch (Exception x)
      {
        harness.debug(x);
        harness.fail("TestOfTiger.testOneMillionA");
      }

    try
      {
        for (int i = 1; i < 128; i++)
          {
            algorithm.update(new byte[i], 0, i);
            byte[] md = algorithm.digest();
            harness.check(ZERO_BITS_ANSWERS[i - 1].equals(Util.toString(md)),
                          "test" + (i * 8) + "ZeroBits");
          }
      }
    catch (Exception x)
      {
        harness.debug(x);
        harness.fail("TestOfTiger.testZeroBits");
      }

    try
      {
        byte[] input = new byte[64];
        input[0] = (byte) 0x80;
        for (int i = 0; i < SINGLE_BIT_ANSWERS.length; i++)
          {
            algorithm.update(input, 0, input.length);
            byte[] md = algorithm.digest();
            harness.check(SINGLE_BIT_ANSWERS[i].equals(Util.toString(md)),
                          "testSingleBit[" + i + "]");
            shiftRight1(input);
          }
      }
    catch (Exception x)
      {
        harness.debug(x);
        harness.fail("TestOfTiger.testZeroBits");
      }

    try
      {
        algorithm = new Tiger();
        algorithm.update("a".getBytes(), 0, 1);
        clone = (IMessageDigest) algorithm.clone();
        byte[] md = algorithm.digest();
        String exp = "77BEFBEF2E7EF8AB2EC8F93BF587A7FC613E247F5F247809";
        harness.check(exp.equals(Util.toString(md)), "testCloning #1");

        clone.update("bc".getBytes(), 0, 2);
        md = clone.digest();
        exp = "2AAB1484E8C158F2BFB8C5FF41B57A525129131C957B5F93";
        harness.check(exp.equals(Util.toString(md)), "testCloning #2");
      }
    catch (Exception x)
      {
        x.printStackTrace();
        harness.debug(x);
        harness.fail("TestOfTiger.testCloning");
      }
  }

  /**
   * Shift, in situ, the variable key/text byte array one position to the
   * right.
   *
   * @param kb The bytes to shift.
   */
  private static void shiftRight1(byte[] kb)
  {
    int i;
    for (i = 0; kb[i] == 0 && i < kb.length; i++)
      { // do nothing
      }
    kb[i] = (byte) ((kb[i] & 0xff) >>> 1);
    // handle byte boundary case
    if (kb[i] == 0)
      {
        i++;
        if (i < kb.length)
          kb[i] = (byte) 0x80;
      }
  }
}